<?php

function bifitkassa_save_logfile($data, $order_number)
{
	$file = dirname(__FILE__) . "/logs/log_" . $order_number . ".txt";

	$top =
		"## " .
		date("Y-m-d H:i:s") .
		" | " .
		t("БИФИТ Касса") .
		" | " .
		t("Заказ №") .
		$order_number;
	$len = mb_strlen($top, "UTF-8");
	$line = str_repeat("#", $len);
	$content = file_exists($file) ? file_get_contents($file) : "";
	$content .= $line . "\n" . $top . "\n" . $line . "\n" . $data . "\n";

	file_put_contents($file, $content);
}

function bifitkassa_yes_no()
{
	return [
		"0" => "Нет",
		"1" => "Да",
	];
}

function bifitkassa_yes_no_not()
{
	return [
		"-1" => "Не учитывать",
		"0" => "Нет",
		"1" => "Да",
	];
}

function bifitkassa_address_type()
{
	return [
		"email" => "Email",
		"phone" => "Телефон",
	];
}

function bifitkassa_calculation_method()
{
	return [
		"PREPAY_FULL" =>
			"Полная предварительная оплата до момента передачи предмета расчёта",
		"PREPAY_PARTIAL" =>
			"Частичная предварительная оплата до момента передачи предмета расчёта",
		"AVANS" => "Аванс",
		"FULL_PAY" =>
			"Полная оплата, в том числе с учётом аванса (предварительной оплаты) в момент передачи предмета расчёта",
		"PARTIAL_SETTLEMENT_AND_CREDIT" =>
			"Частичная оплата предмета расчёта в момент его передачи с последующей оплатой в кредит",
		"TRANSFER_ON_CREDIT" =>
			"Передача предмета расчёта без его оплаты в момент его передачи с последующей оплатой в кредит",
		"CREDIT_PAYMENT" =>
			"Оплата предмета расчёта после его передачи с оплатой в кредит (оплата кредита)",
	];
}

function bifitkassa_payment_type()
{
	return [
		"CASH" => "Сумма оплаты наличными",
		"CARD" => "Сумма оплата безналичными",
		"PREPAY" => "Сумма оплаты предоплатой (зачётом аванса)",
		"POSTPAY" => "Сумма оплаты постоплатой (в кредит)",
		"OTHER" => "Сумма оплаты встречным предоставлением",
	];
}

function bifitkassa_vat()
{
	return [
		"WITHOUT_VAT" => "Без НДС",
		"VAT_0" => "НДС 0%",
		"VAT_5" => "НДС 5%",
		"VAT_7" => "НДС 7%",
		"VAT_10" => "НДС 10%",
		"VAT_20" => "НДС 20%",
		"VAT_22" => "НДС 22%",
		"VAT_105" => "НДС 5/105",
		"VAT_107" => "НДС 7/107",
		"VAT_110" => "НДС 10/110",
		"VAT_120" => "НДС 20/120",
		"VAT_122" => "НДС 22/122",
	];
}

function bifitkassa_tax_system()
{
	return [
		"COMMON" => "ОСН",
		"SIMPLIFIED" => "УСН доход",
		"SIMPLIFIED_WITH_EXPENSE" => "УСН доход-расход",
		"ENVD" => "ЕНВД",
		"COMMON_AGRICULTURAL" => "ЕСХН",
		"PATENT" => "ПАТЕНТ",
	];
}

function bifitkassa_pm_ids()
{
	$query = db_select("rules_config", "r");
	$query->fields("r", ["id", "name", "label", "module"]);
	$query->condition("module", "commerce_payment");
	$results = $query->execute();
	$rows = [];

	foreach ($results as $key => $row) {
		$rows[$row->name] = $row->label;
	}

	return $rows;
}

function bifitkassa_deliveries()
{
	$query = db_select("commerce_flat_rate_service", "s");
	$query->fields("s", ["name", "title"]);
	$results = $query->execute();
	$rows = [];

	foreach ($results as $key => $row) {
		$rows[$row->name] = $row->title;
	}

	return $rows;
}

function bifitkassa_deliveries_count($c = 2)
{
	$d = bifitkassa_deliveries();

	if (count($d) > 0) {
		$c = count($d);
	}

	return $c;
}

function bifitkassa_pm_statuses()
{
	return [
		"Canceled" => [
			"canceled" => t("Canceled"),
		],
		"Shopping cart" => [
			"cart" => t("Shopping cart"),
		],
		"Checkout" => [
			"checkout_checkout" => t("Checkout:") . " " . t("Checkout"),
			"checkout_shipping" => t("Checkout:") . " " . t("Shipping"),
			"checkout_review" => t("Checkout:") . " " . t("Review"),
			"checkout_payment" => t("Checkout:") . " " . t("Payment"),
			"checkout_complete" => t("Checkout:") . " " . t("Complete"),
		],
		"Pending" => [
			"pending" => t("Pending"),
			"processing" => t("Processing"),
		],
		"Completed" => [
			"completed" => t("Completed"),
		],
	];
}

function bifitkassa_display_date($time)
{
	return isset($time)
		? format_date($time, "custom", COMMERCE_BIFITKASSA_FORMAT_DATE)
		: "";
}

function bifitkassa_order_link($id)
{
	return '<a href="/admin/commerce/orders/' .
		$id .
		'" class="button" style="margin:auto;">#' .
		$id .
		"</a>";
}

function bifitkassa_receipt_link($url)
{
	return $url == "XXX"
		? '<span class="error">' . t("нет") . "</span>"
		: '<a target="_blank" href="' .
				$url .
				'" class="button" style="margin:auto;">' .
				t("Открыть") .
				"</a>";
}

function bifitkassa_receipt_id($id)
{
	$html =
		$id > 0
			? '<span class="success">' . $id . "</span>"
			: '<span class="error">' . t("нет") . "</span>";

	return $html;
}

function bifitkassa_log_link($log, $key)
{
	$html =
		'<a href="javascript:void(0);" id="opener' .
		$key .
		'" class="button" style="margin:auto;">' .
		t("Смотреть") .
		"</a>";
	$html .=
		'<div id="dialog' .
		$key .
		'" title="' .
		t("Лог-файл") .
		'"><p>' .
		nl2br($log) .
		"</p></div>";
	$html .= "<script>";
	$html .=
		'jQuery("#dialog' .
		$key .
		'").dialog({
				modal:true,
				autoOpen:false
			});';
	$html .= 'jQuery("#opener' . $key . '").click(function(){';
	$html .= 'jQuery("#dialog' . $key . '").dialog("open");';
	$html .= "});";
	$html .= "</script>";

	return $html;
}

function bifitkassa_receipt_type($type, $id)
{
	$label = "COMMERCE_BIFITKASSA_TYPE_" . strtoupper($type);
	$html =
		'<span class="type type-' .
		((int) $id > 0 ? "success" : "error") .
		" type-" .
		strtolower($type) .
		'">' .
		constant($label) .
		"</span>";

	return $html;
}

function commerce_bifitkassa_create_guid($order)
{
	// Create GUID (Globally Unique Identifier)
	$guid = "";
	$namespace = rand(11111, 99999);
	$uid = uniqid("", true);
	$data = $namespace;
	$data .= $_SERVER["REQUEST_TIME"];
	$data .= $_SERVER["HTTP_USER_AGENT"];
	$data .= $_SERVER["REMOTE_ADDR"];
	$data .= $_SERVER["REMOTE_PORT"];
	$data .= $order["number"];
	$data .= $order["summ"];
	$hash = strtoupper(hash("ripemd128", $uid . $guid . md5($data)));
	$guid =
		substr($hash, 0, 8) .
		"-" .
		substr($hash, 8, 4) .
		"-" .
		substr($hash, 12, 4) .
		"-" .
		substr($hash, 16, 4) .
		"-" .
		substr($hash, 20, 12);

	return strtolower($order["cms"] . "-" . $guid);
}

function commerce_bifitkassa_check_price(
	$item_price,
	$order_subtotal,
	$order_discount = 0
) {
	// if the discount is more than zero
	if ($order_discount > 0) {
		// calculate the discount percentage
		$percent = ($order_discount * 100) / $order_subtotal;

		// reduce the price of goods according to the discount
		$item_price = $item_price - $item_price * ($percent / 100);
	}

	return $item_price;
}

function commerce_bifitkassa_check_phone($tel)
{
	// check the fullness
	if (isset($tel) && trim($tel) != "") {
		// remove extra characters
		$phone = str_replace(["+", "(", ")", "-", " "], "", trim($tel));

		// get the first digit
		$first = substr($phone, "0", 1);

		// if the number is 10 digits and the first digit is not 7, then add 7
		$phone = strlen($phone) == 10 && $first != 7 ? "7" . $phone : $phone;

		// Final inspection
		$phone = !preg_match("/^[0-9]{11}+$/", $phone)
			? t("Номер " . $phone . " некорректен!")
			: $phone;
	} else {
		// The phone is empty or not found!
		$phone = t("Номер телефона пуст или не найден!");
	}

	return $phone;
}

function commerce_bifitkassa_check_email($email)
{
	// check fullness
	if (isset($email) && trim($email) != "") {
		// record model check
		$email = !preg_match("/^\+\d+$|^\S+@\S+$/", $email)
			? t("E-mail " . $email . " некорректен!")
			: $email;
	} else {
		// Email is empty or not found!
		$email = t("E-mail пуст или не найден!");
	}

	return $email;
}

function commerce_bifitkassa_check_inn($inn, $type = 1)
{
	// check the fullness
	if (isset($inn) && trim($inn) != "") {
		if ($type == 2) {
			// only numbers in the amount of 10 pcs.
			$inn = !preg_match("/^\d{10}$/", trim($inn))
				? t("Некорректный ИНН!")
				: trim($inn);
		} elseif ($type == 1) {
			// only numbers in the amount of 12 pcs.
			$inn = !preg_match("/^\d{12}$/", trim($inn))
				? t("Некорректный ИНН!")
				: trim($inn);
		} else {
			// only numbers in the amount of 10 or 12 pieces.
			$inn = !preg_match("/^\d{10}$|^\d{12}$/", trim($inn))
				? t("Некорректный ИНН!")
				: trim($inn);
		}
	} else {
		// TIN is empty or not found!
		$inn = t("ИНН не задан!");
	}

	return $inn;
}

function commerce_bifitkassa_save_report(
	$test,
	$bifit,
	$receipt,
	$receipt_id,
	$doc,
	$object,
	$idemp_key
) {
	// display the property of the operation
	$log =
		$test != ""
			? "####### " . t("Тестовый запрос") . "\n"
			: "####### " . t("Боевой запрос") . "\n";

	// set the time zone according to the site settings
	$log .= "####### " . date("Y-m-d H:i:s", time()) . "\n";

	// ANSWER #1

	$log .= "####### " . t("Ответ") . " #1: " . "\n";

	// response data # 1
	$token_error = null;

	if (is_object($bifit)) {
		foreach ($bifit as $key => $val) {
			if ($key == "error") {
				$token_error = 1;
			}
			$log .= $key . ": " . $val . "\n";
		}
	}

	// RECEIPT

	// generated receipt for sending
	$log .= "####### " . t("Чек") . ": " . "\n" . $receipt . "\n";

	if (!$token_error) {
		// IDEMPOTENCY KEY

		$log .=
			"####### " .
			t("Ключ идемпотентности") .
			": " .
			"\n" .
			$idemp_key .
			"\n";

		// ANSWER #2

		$log .= "####### " . t("Ответ") . " #2: " . "\n";

		// response data #2
		if (is_object($receipt_id)) {
			$log .= commerce_bifitkassa_check_responce($receipt_id);
		} else {
			$log .= t("Номер документа") . ": " . $receipt_id . "\n";

			// ANSWER #3

			$log .= "#######  " . t("Ответ") . " #3: " . "\n";

			if (is_object($doc)) {
				$log .= commerce_bifitkassa_check_responce($doc);
			}

			// LINK TO THE CHECK

			// generated link to the check
			$log .=
				"####### " .
				t("Ссылка на чек") .
				": " .
				"\n" .
				$object->receipt_link;
		}
	}

	$object->log = $log;

	// SAVE IN DB

	$insert_fields = [
		"order_id" => $object->order_id,
		"order_number" => $object->order_number,
		"order_date" => $object->order_date,
		"action_date" => $object->action_date,
		"receipt_link" => $object->receipt_link,
		"receipt_id" => $object->receipt_id,
		"receipt_type" => $object->receipt_type,
		"log" => $object->log,
	];

	db_insert("commerce_bifitkassa_reports")
		->fields($insert_fields)
		->execute();
}

function commerce_bifitkassa_get_amount($value, $get_value = true)
{
	$amount = $value->amount;

	if ($get_value) {
		$amount = $amount->value();
	}

	return (float) number_format($amount / 100, 2, ".", "");
}

function commerce_bifitkassa_get_quantity($value)
{
	return (float) number_format($value->quantity->value(), 1);
}

function commerce_bifitkassa_get_tax_rate($line_item_wrapper)
{
	$rate = 0;
	$price = $line_item_wrapper->commerce_unit_price->value();

	if (isset($price["data"]["components"][1]["price"]["data"]["tax_rate"])) {
		$rate =
			$price["data"]["components"][1]["price"]["data"]["tax_rate"][
				"rate"
			];
	}

	return (int) $rate * 100;
}

function commerce_bifitkassa_get_data(
	$url,
	$headers = false,
	$data = false,
	$type = "POST"
) {
	// initialize the cURL session
	$ch = curl_init($url);

	// assign request type
	if ($type != "POST") {
		curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $type);
	} else {
		curl_setopt($ch, CURLOPT_POST, true);
	}

	// add data
	if ($data) {
		curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
	}

	// returning the result as a string
	curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
	curl_setopt($ch, CURLOPT_ENCODING, "");
	curl_setopt($ch, CURLOPT_MAXREDIRS, 10);
	curl_setopt($ch, CURLOPT_TIMEOUT, 0);
	curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
	curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);

	// add headers
	if ($headers) {
		curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
	}

	// get the answer
	$output = curl_exec($ch);

	// close the cURL session
	curl_close($ch);

	return $output;
}

function commerce_bifitkassa_check_responce($response)
{
	$mess = "";

	if (
		is_object($response) &&
		($response->type == "ERROR" ||
			$response->status == 400 ||
			$response->status == 500)
	) {
		$mess = "";

		if ($response->status == 400 || $response->status == 500) {
			$mess .= $response->error . "\n";
			$mess .= $response->message . "\n";
		} else {
			$array = json_decode(json_encode($response), true);
			array_walk_recursive($array, function ($item, $key) use (&$mess) {
				if ($item != "ERROR" && trim($item != "")) {
					$mess .= $key . ": " . $item . "\n";
				}
			});
		}
	} elseif (is_object($response)) {
		$array = json_decode(json_encode($response), true);
		array_walk_recursive($array, function ($item, $key) use (&$mess) {
			if (trim($item != "")) {
				$mess .= $key . ": " . $item . "\n";
			}
		});
	}

	return $mess;
}

function commerce_bifitkassa_get_token($login, $password)
{
	$signature = base64_encode(hash("sha256", $password, true));
	$data = "username=" . $login;
	$data .= "&password=" . $signature;
	$data .= "&client_id=cashdesk-rest-client";
	$data .= "&client_secret=cashdesk-rest-client";
	$data .= "&grant_type=password";
	$url = "https://kassa.bifit.com/cashdesk-api/v1/oauth/token";
	$headers = ["Content-Type: application/x-www-form-urlencoded"];
	$tokenData = json_decode(
		commerce_bifitkassa_get_data($url, $headers, $data)
	);

	return $tokenData;
}

function commerce_bifitkassa_get_organizations($token)
{
	$url =
		"https://kassa.bifit.com/cashdesk-api/v1/protected/organizations/list/read";
	$headers = ["Authorization: Bearer " . $token];
	$orgData = json_decode(commerce_bifitkassa_get_data($url, $headers));

	return $orgData;
}

function commerce_bifitkassa_get_tradeObjects($org_id, $token)
{
	$data = "organization_id=" . $org_id;
	$url =
		"https://kassa.bifit.com/cashdesk-api/v1/protected/trade_objects/list/read";
	$headers = [
		"Content-Type: application/x-www-form-urlencoded",
		"Authorization: Bearer " . $token,
	];
	$tradeData = json_decode(
		commerce_bifitkassa_get_data($url, $headers, $data)
	);

	return $tradeData;
}

function commerce_bifitkassa_create_main_array($orgData, $token)
{
	$arr = [];

	foreach ($orgData as $key => $org) {
		$arr[$key] = new stdClass();
		$arr[$key]->id = $org->id;
		$arr[$key]->name = $org->name;
		$arr[$key]->traides = [];

		$tradeData = commerce_bifitkassa_get_tradeObjects($org->id, $token);

		foreach ($tradeData as $k => $trd) {
			$arr[$key]->traides[$k] = new stdClass();
			$arr[$key]->traides[$k]->id = $trd->id;
			$arr[$key]->traides[$k]->name = $trd->name;
		}
	}

	return $arr;
}

function commerce_bifitkassa_create_selects($array, $name)
{
	$html =
		'<select id="edit-' .
		str_replace("_", "-", $name) .
		'" name="' .
		$name .
		'" class="form-select">';

	foreach ($array as $key => $val) {
		$html .= '<option value="' . $key . '">';
		$html .= $val;
		$html .= "</option>";
	}

	$html .= "</select>";

	return $html;
}

function commerce_bifitkassa_get_LkData($order, $i)
{
	if (empty($order->order_id)) {
		return;
	}

	// params
	$wrapper = entity_metadata_wrapper("commerce_order", $order->order_id);

	// tax system
	$tax_system = variable_get(
		"commerce_bifitkassa_tax_system_" . $i,
		"SIMPLIFIED_WITH_EXPENSE"
	);
	// method of calculation
	$calc_method = variable_get(
		"commerce_bifitkassa_calculation_method",
		"FULL_PAY"
	);
	// uniform VAT
	$vat_mode = variable_get("commerce_bifitkassa_vat_mode", "1");
	// VAT
	$vat = variable_get("commerce_bifitkassa_vat", "WITHOUT_VAT");
	// client address type
	$address_type = variable_get("commerce_bifitkassa_address_type", "email");

	// user
	$default_pid = commerce_addressbook_get_default_profile_id(
		$order->uid,
		"shipping"
	);
	$profile_load = commerce_customer_profile_load($default_pid);
	$profile = $profile_load->commerce_customer_address["und"][0];

	$name_line = $profile["name_line"];
	$user_first_name = $profile["first_name"];
	$user_last_name = $profile["last_name"];
	$user_email = commerce_bifitkassa_check_email($wrapper->mail->value());
	$user_phone = commerce_bifitkassa_check_phone(
		$profile_load->field_phone["und"][0]["value"]
	);

	// address
	$address = $address_type == "email" ? $user_email : $user_phone;
	$user_address = "";

	if (isset($profile["thoroughfare"]) && $profile["thoroughfare"] != "") {
		$user_address .= $profile["thoroughfare"];
	}

	if (isset($profile["locality"]) && $profile["locality"] != "") {
		$user_address .= $user_address != "" ? ", " : "";
		$user_address .= $profile["locality"];
	}

	if (isset($profile["postal_code"]) && $profile["postal_code"] != "") {
		$user_address .= $user_address != "" ? ", " : "";
		$user_address .= $profile["postal_code"];
	}

	if (
		isset($profile["administrative_area"]) &&
		$profile["administrative_area"] != ""
	) {
		$user_address .= $user_address != "" ? ", " : "";
		$user_address .= $profile["administrative_area"];
	}

	if (isset($profile["country"]) && $profile["country"] != "") {
		$user_address .= $user_address != "" ? ", " : "";
		$user_address .= $profile["country"];
	}

	$user_address = $user_address != "" ? $user_address . "." : $user_address;

	// order data
	/// order id
	$order_id = (int) $order->order_id;
	/// order number
	$order_number = $order->order_number;
	/// order created
	$order_created = $order->changed;
	/// order changed
	$order_changed = $order->changed;
	/// order number
	$order_number = $order->order_number;
	/// order total
	$commerce_order_total = $wrapper->commerce_order_total->value();
	$order_total = (float) number_format(
		$commerce_order_total["amount"] / 100,
		2,
		".",
		""
	);
	/// order payment
	$payments = commerce_payment_transaction_load_multiple(
		[],
		["order_id" => $order->order_id]
	);
	$payment = !empty($payments) ? array_shift($payments) : null;
	$order_payed = $payment->status == "success" ? 1 : null;
	$order_payed_date = $payment->changed
		? $payment->changed
		: $payment->created;

	// order payment items
	$o_subtotal = 0;
	$o_discount = 0;
	$o_shipment = 0;
	$o_total = 0;

	$order_delivery = null;
	$order_delivery_price = 0;

	// array parameters
	/// an array of products to fill
	$r_products = [];

	// key of the array of goods for the receipt
	$key_id = 0;

	foreach ($wrapper->commerce_line_items as $key => $line_item_wrapper) {
		$line_item_type = $line_item_wrapper->getBundle();
		$item_sum = commerce_bifitkassa_get_amount(
			$line_item_wrapper->commerce_total
		);
		$item_price = commerce_bifitkassa_get_amount(
			$line_item_wrapper->commerce_unit_price
		);
		$item_quantity = commerce_bifitkassa_get_quantity($line_item_wrapper);

		if (in_array($line_item_type, commerce_product_line_item_types())) {
			$item_name = $line_item_wrapper->commerce_product->value()->title;
		} else {
			$item_name = $line_item_wrapper->line_item_label->value();
		}

		if ($line_item_type == "product") {
			// we clear the name of the product from special characters
			$p_name = preg_replace(
				"/[^\p{L}0-9 ]/iu",
				" ",
				html_entity_decode($item_name)
			);
			$p_name = preg_replace("/ +/", " ", trim($p_name));

			// product codes
			$item_id = $line_item_wrapper->commerce_product->value()
				->product_id;
			$item_ean = $line_item_wrapper->line_item_label->value();

			// tax_rate
			$tax_rate = commerce_bifitkassa_get_tax_rate($line_item_wrapper);

			// form an array of goods for the check
			$r_products[$key_id] = new stdClass();
			$r_products[$key_id]->product_id = $item_id;
			$r_products[$key_id]->product_name = $p_name;
			$r_products[$key_id]->product_item_price = $item_price;
			$r_products[$key_id]->product_quantity = $item_quantity;
			$r_products[$key_id]->product_full_price = $item_sum;
			$r_products[$key_id]->product_vendor_code = $item_ean;
			$r_products[$key_id]->product_tax_value = $tax_rate;
			$key_id++;

			$o_subtotal += $item_sum;
		}

		if ($line_item_type == "commerce_discount") {
			$o_discount += $item_sum < 0 ? abs($item_sum) : $item_sum;
		}

		if ($line_item_type == "shipping") {
			$order_delivery = $line_item_wrapper->commerce_shipping_service->value();
			$order_delivery_price = $item_sum;
			$order_delivery_tax_rate = commerce_bifitkassa_get_tax_rate(
				$line_item_wrapper
			);
		}

		$o_total += $item_sum;
	}

	// lk data
	$lk = new stdClass();
	$lk->order_id = $order_id;
	$lk->order_number = $order_number;
	$lk->order_created = $order_created;
	$lk->order_changed = $order_changed;
	$lk->order_subtotal = $o_subtotal;
	$lk->order_discount = $o_discount;
	$lk->order_price = $order_total;
	$lk->order_delivery_id = $order_delivery;
	$lk->order_price_delivery = $order_delivery_price;
	$lk->order_delivery_tax_value = $order_delivery_tax_rate;
	$lk->order_status_id = $order->status;
	$lk->order_payed = $order_payed;
	$lk->order_date_payed = $order_payed_date;
	$lk->user_name = $user_first_name;
	$lk->user_last_name = $user_last_name;
	$lk->user_email = $user_email;
	$lk->user_phone = $user_phone;
	$lk->user_address = $user_address;
	$lk->address = $address;
	$lk->r_products = $r_products;
	$lk->tax_system = $tax_system;
	$lk->calc_method = $calc_method;
	$lk->vat_mode = $vat_mode;
	$lk->vat = $vat;
	$lk->vat_value = str_replace(["WITHOUT_VAT", "VAT_"], ["0", ""], $lk->vat);

	return $lk;
}

function commerce_bifitkassa_save_order($lk)
{
	// Options
	$login = commerce_bifitkassa_check_phone(
		variable_get("commerce_bifitkassa_login", null)
	);
	$password = variable_get("commerce_bifitkassa_password", null);
	$organization_id = variable_get("commerce_bifitkassa_organization");
	$trade_object_id = variable_get("commerce_bifitkassa_trade_object");
	$use_refused = variable_get("commerce_bifitkassa_lk_use_refused", null);

	// log
	$logData = "## " . t("Получаем токен...") . "\n";

	// Get token
	$tokenData = commerce_bifitkassa_get_token($login, $password);
	$token = $tokenData->access_token;

	// log
	if ($token) {
		$logData .= "## " . t("Получен токен") . ": " . $token . "\n";
	} else {
		$logData .=
			"## " . t("В процессе получения токена произошла ошибка!") . "\n";
		$logData .=
			"## " .
			t("Ошибка") .
			": " .
			$tokenData->error .
			"\n" .
			"## " .
			t("Описание ошибки") .
			": " .
			$tokenData->error_description;
		// save log
		bifitkassa_save_logfile($logData, $lk->order_number);

		return false;
	}

	// parameters for ordering
	$onlineOrderItems = "";

	// log
	$logData .=
		"## " .
		t('Проверяем наличие товаров заказа в базе "БИФИТ Касса"...') .
		"\n";

	// form order data
	foreach ($lk->r_products as $key_id => $product) {
		// address for POST request
		$url =
			"https://kassa.bifit.com/cashdesk-api/v1/protected/nomenclatures/external_id/" .
			$product->product_id .
			"?organization_id=" .
			$organization_id;

		// link data
		$data = false;

		// headers
		$headers = [
			"Content-Type: application/x-www-form-urlencoded",
			"Authorization: Bearer " . $token,
		];

		$prod = json_decode(
			commerce_bifitkassa_get_data($url, $headers, $data, "GET")
		);

		// correct price
		$product->product_item_price = commerce_bifitkassa_check_price(
			$product->product_item_price,
			$lk->order_subtotal,
			$lk->order_discount
		);

		if ($prod == null) {
			// CREATE A NEW PRODUCT

			// log
			$logData .=
				"## " .
				t("Товар с Id=") .
				$product->product_id .
				" " .
				t('не найден в базе "БИФИТ Касса".') .
				"\n";
			$logData .=
				"## " .
				t('Добавляем новый товар в базу "БИФИТ Касса"...') .
				"\n";

			// address for POST request
			$url =
				"https://kassa.bifit.com/cashdesk-api/v1/protected/nomenclatures";

			// link data
			$data = "{";
			$data .= '"item":{';
			$data .=
				'"vatValue":"' .
				($lk->vat_mode == 1
					? $lk->vat_value
					: $product->product_tax_value) .
				'",';
			$data .= '"vendorCode":"' . $product->product_vendor_code . '",';
			$data .= '"name":"' . $product->product_name . '",';
			$data .= '"externalId":"' . $product->product_id . '",';
			$data .= '"sellingPrice":' . $product->product_item_price;
			$data .= "},";
			$data .= '"organizationId":"' . $organization_id . '"';
			$data .= "}";

			// headers
			$headers = [
				"Content-Type: application/json",
				"Authorization: Bearer " . $token,
			];

			// log
			$logData .=
				t('Данные для отправки в базу "БИФИТ Касса"') . ":" . "\n";
			$logData .= $data . "\n";

			// get data from request
			$tov = json_decode(
				commerce_bifitkassa_get_data($url, $headers, $data)
			);

			// log
			if (!is_object($tov) && !is_array($tov)) {
				$logData .=
					"## " .
					t(
						'Новый товар успешно добавлен в базу "БИФИТ Касса" с Id='
					) .
					$tov .
					"\n";
			} else {
				$logData .=
					"## " .
					t("В процессе добавления нового товара произошла ошибка") .
					":";
				$logData .= commerce_bifitkassa_check_responce($tov) . "\n";
			}
		} else {
			$tov = $prod->id;

			// log
			$logData .=
				"## " .
				t("Товар с Id=") .
				$product->product_id .
				" " .
				t('уже существует в базе "БИФИТ Касса" с Id=') .
				$tov .
				"\n";
		}

		// display product data in the order
		$onlineOrderItems .= $onlineOrderItems != "" ? ", " : "";
		$onlineOrderItems .= "{";
		$onlineOrderItems .= '"nomenclatureId":"' . $tov . '",';
		$onlineOrderItems .= '"paymentSubject":"PRODUCT",';
		$onlineOrderItems .= '"name":"' . $product->product_name . '",';
		$onlineOrderItems .= '"price":' . $product->product_item_price . ",";
		$onlineOrderItems .= '"quantity":' . $product->product_quantity;
		$onlineOrderItems .= "}";
	}

	// delivery details
	if ($lk->order_price_delivery > 0) {
		// address for POST request
		$url =
			"https://kassa.bifit.com/cashdesk-api/v1/protected/nomenclatures/external_id/" .
			"delivery_" .
			$lk->order_delivery_id .
			"?organization_id=" .
			$organization_id;

		// link data
		$data = false;

		// headers
		$headers = [
			"Content-Type: application/x-www-form-urlencoded",
			"Authorization: Bearer " . $token,
		];

		$delivery = json_decode(
			commerce_bifitkassa_get_data($url, $headers, $data, "GET")
		);

		// log
		$logData .=
			"## " .
			t('Проверяем наличие услуги "Доставка" в базе "БИФИТ Касса"...') .
			"\n";

		if ($delivery == null) {
			// CREATE A NEW SERVICE

			// log
			$logData .=
				"## " .
				t('Услуга "Доставка" не найдена в базе "БИФИТ Касса".') .
				"\n";
			$logData .=
				"## " .
				t('Добавляем новую услугу "Доставка" в базу "БИФИТ Касса"...') .
				"\n";

			// address for POST request
			$url =
				"https://kassa.bifit.com/cashdesk-api/v1/protected/nomenclatures";

			// link data
			$data = "{";
			$data .= '"item":{';
			$data .=
				'"vatValue":"' .
				($lk->vat_mode == 1
					? $lk->vat_value
					: $lk->order_delivery_tax_value) .
				'",';
			$data .= '"name":"' . t("Доставка") . '",';
			$data .=
				'"externalId":"' . "delivery_" . $lk->order_delivery_id . '",';
			$data .= '"sellingPrice":' . $lk->order_price_delivery;
			$data .= "},";
			$data .= '"organizationId":"' . $organization_id . '"';
			$data .= "}";

			// headers
			$headers = [
				"Content-Type: application/json",
				"Authorization: Bearer " . $token,
			];

			// log
			$logData .=
				t('Данные для отправки в базу "БИФИТ Касса"') . ":" . "\n";
			$logData .= $data . "\n";

			// get data from request
			$tov = json_decode(
				commerce_bifitkassa_get_data($url, $headers, $data)
			);

			// log
			if (!is_object($tov) && !is_array($tov)) {
				$logData .=
					"## " .
					t(
						'Новая услуга "Доставка" успешно добавлена в базу "БИФИТ Касса" с Id='
					) .
					$tov .
					"\n";
			} else {
				$logData .=
					"## " .
					t(
						'В процессе добавления услуги "Доставка" произошла ошибка:'
					) .
					"\n";
				$logData .= commerce_bifitkassa_check_responce($tov) . "\n";
			}
		} else {
			$tov = $delivery->id;

			// log
			$logData .=
				"## " .
				t(
					'Услуга "Доставка" уже существует в базе "БИФИТ Касса" c Id='
				) .
				$tov .
				"\n";
		}

		// display data of delivery method in order
		$onlineOrderItems .= $onlineOrderItems != "" ? ", " : "";
		$onlineOrderItems .= "{";
		$onlineOrderItems .= '"nomenclatureId":"' . $tov . '", ';
		$onlineOrderItems .= '"paymentSubject":"SERVICE", ';
		$onlineOrderItems .= '"name":"' . t("Доставка") . '", ';
		$onlineOrderItems .= '"price":' . $lk->order_price_delivery . ", ";
		$onlineOrderItems .= '"quantity":1';
		$onlineOrderItems .= "}";
	}

	// CREATE A USER IN THE CASH

	// address for POST request
	$url = "https://kassa.bifit.com/cashdesk-api/v1/protected/clients";

	// headers
	$headers = [
		"Content-Type: application/json",
		"Authorization: Bearer " . $token,
	];

	// link data
	$data = "{";
	$data .= '"organizationId":"' . $organization_id . '",';
	$data .= '"address":"' . $lk->user_address . '",';
	$data .= '"lastName":"' . trim($lk->user_last_name) . '",';
	$data .= '"firstName":"' . trim($lk->user_name) . '",';
	$data .= '"phone":"' . $lk->user_phone . '",';
	$data .= '"email":"' . $lk->user_email . '"';
	$data .= "}";

	// log
	$logData .=
		"## " .
		t('Пробуем добавить нового клиента в базу "БИФИТ Касса"...') .
		"\n";
	$logData .=
		"## " . t('Данные для отправки в базу "БИФИТ Касса"') . ":" . "\n";
	$logData .= $data . "\n";

	// trying to add a new client
	$clientId = json_decode(
		commerce_bifitkassa_get_data($url, $headers, $data)
	);

	// such a client is already in the database
	if ($clientId->type == "ERROR") {
		// address for POST request
		$url =
			"https://kassa.bifit.com/cashdesk-api/v1/protected/clients/list/read?organization_id=" .
			$organization_id;

		// link data
		$data = false;

		// headers
		$headers = [
			"Content-Type: application/json",
			"Authorization: Bearer " . $token,
		];

		// get all clients
		$clients = json_decode(
			commerce_bifitkassa_get_data($url, $headers, $data)
		);

		foreach ($clients as $clientsItem) {
			// phone number match
			if ($clientsItem->phone == $lk->user_phone) {
				$clientId = $clientsItem->id;
			}
		}

		// log
		$logData .=
			"## " .
			t('Текущий клиент уже существует в базе "БИФИТ Касса" с Id=') .
			$clientId .
			"\n";
	} else {
		// log
		$logData .=
			"## " .
			t('Добавлен новый клиент в базу "БИФИТ Касса" с Id=') .
			$clientId .
			"\n";
	}

	// SENDING AN ORDER TO THE CASHIER

	// checking order availability
	// address for GET request
	$url =
		"https://kassa.bifit.com/cashdesk-api/v1/protected/online_orders/list/read/" .
		$clientId .
		"?organization_id=" .
		$organization_id;

	// link data
	$data = false;

	// headers
	$headers = [
		"Content-Type: application/x-www-form-urlencoded",
		"Authorization: Bearer " . $token,
	];

	// get orders
	$clientOrders = json_decode(
		commerce_bifitkassa_get_data($url, $headers, $data, "GET")
	);

	$order_deleted = null;
	$receipt_id = 0;
	$del_type = "";

	// log
	$logData .=
		"## " .
		t('Проверяем наличие текущего заказа в базе "БИФИТ Касса"...') .
		"\n";

	// trans
	$l = [
		"COURIER" => t("Курьер (COURIER)"),
		"SELF" => t("Самовывоз (SELF)"),
	];

	// search order by id
	foreach ($clientOrders as $key => $clientOrder) {
		if ($clientOrder->externalId == $lk->order_number) {
			$receipt_id = $clientOrder->id;
			$del_type = $clientOrder->deliveryType;

			// log
			$logData .=
				"## " .
				t('Текущий заказ уже существует в базе "БИФИТ Касса" с Id=') .
				$receipt_id .
				". " .
				t("Тип доставки=") .
				$l[$del_type] .
				"." .
				"\n";

			break;
		}
	}

	// log
	if (!$receipt_id) {
		$logData .=
			"## " . t('Текущий заказ не найден в базе "БИФИТ Касса".') . "\n";
	}

	// type of delivery
	$lk_courier_deliveries = variable_get(
		"commerce_bifitkassa_lk_courier_delivery"
	);
	$lk_self_deliveries = variable_get("commerce_bifitkassa_lk_self_delivery");

	$deliveryType = "SERVICE";

	if (
		isset($lk_courier_deliveries) &&
		in_array($lk->order_delivery_id, $lk_courier_deliveries)
	) {
		$deliveryType = "COURIER";
	} elseif (
		isset($lk_self_deliveries) &&
		in_array($lk->order_delivery_id, $lk_self_deliveries)
	) {
		$deliveryType = "SELF";
	}

	// assign "is_update"
	$lk_refused_statuses = variable_get(
		"commerce_bifitkassa_lk_refused_statuses"
	);
	$is_update =
		$receipt_id > 0 &&
		isset($lk_refused_statuses) &&
		in_array($lk->order_status_id, $lk_refused_statuses) &&
		$use_refused
			? 1
			: null;

	// new order || "is_update" conditions
	if ($deliveryType != "SERVICE" && ($receipt_id == 0 || $is_update)) {
		// address for request
		$url =
			"https://kassa.bifit.com/cashdesk-api/v1/protected/online_orders";

		// headers
		$headers = [
			"Content-Type: application/json",
			"Authorization: Bearer " . $token,
		];

		// data
		if ($is_update) {
			// log
			$logData .=
				"## " .
				t(
					'Начинаем процесс обновления заказа в базе "БИФИТ Касса"...'
				) .
				"\n";

			$url .= "/" . $receipt_id . "/status";
			$type = "PUT";
			$statusType = $del_type == "SELF" ? "REFUSED" : "DELETED";
			// Basic data
			$data = "{";
			$data .= '"onlineOrderId":"' . $receipt_id . '",';
			$data .= '"statusType":"' . $statusType . '",';
			$data .= '"statusDate":0,';
			$data .= '"statusInfo":""';
			$data .= "}";
		} else {
			// log
			$logData .=
				"## " .
				t(
					'Начинаем процесс сохранения нового заказа в базу "БИФИТ Касса"...'
				) .
				"\n";

			// data
			$type = "POST";
			$statusType =
				$deliveryType == "SELF"
					? "ACCEPTED"
					: strtoupper(
						variable_get(
							"commerce_bifitkassa_lk_new_courier_order_status",
							"NEW"
						)
					);
			// Basic data
			$onlineOrder = "{";
			$onlineOrder .= '"organizationId":"' . $organization_id . '",';
			$onlineOrder .= '"tradeObjectId":"' . $trade_object_id . '",';
			$onlineOrder .= '"taxSystem":"' . $lk->tax_system . '",';
			$onlineOrder .= '"externalId":"' . $lk->order_number . '",';
			$onlineOrder .= '"deliveryType":"' . $deliveryType . '",';
			$onlineOrder .=
				'"deliveryAmount":"' . $lk->order_price_delivery . '",';
			$onlineOrder .= '"paid":false,';
			$onlineOrder .= '"orderTime":' . $lk->order_created . ",";
			$onlineOrder .= '"clientId":' . $clientId . ", ";
			$onlineOrder .= '"address":"' . $lk->user_address . '",';
			$onlineOrder .= '"comment":"", ';
			$onlineOrder .= '"responsiblePersonLogin":"' . $login . '",';
			$onlineOrder .= '"totalAmount":' . $lk->order_price . ",";
			$onlineOrder .= '"currentStatusType":"' . $statusType . '",';
			$onlineOrder .= '"currentStatusTime":0';
			$onlineOrder .= "}";

			$data = "{";
			$data .= '"onlineOrder":' . $onlineOrder . ",";
			$data .= '"onlineOrderItems":[' . $onlineOrderItems . "]";
			$data .= "}";
		}

		// log
		$logData .=
			"## " .
			t('Данные заказа для отправки в базу "БИФИТ Касса"') .
			":" .
			"\n";
		$logData .= $data . "\n";

		// get the ID of the document being processed (id)
		$receipt_id = json_decode(
			commerce_bifitkassa_get_data($url, $headers, $data, $type)
		);

		// log
		if (!is_object($receipt_id) && !is_array($receipt_id)) {
			if ($is_update && $statusType == "DELETED") {
				$order_deleted = 1;
				$logData .=
					"## " .
					t('Заказ успешно удалён из базы "БИФИТ Касса".') .
					$receipt_id .
					"\n";
			} elseif ($is_update && $statusType == "REFUSED") {
				$logData .=
					"## " .
					t('Заказ успешно отменён в базе "БИФИТ Касса".') .
					$receipt_id .
					"\n";
			} else {
				$logData .=
					"## " .
					t('Заказ успешно сохранён в базу "БИФИТ Касса" с Id=') .
					$receipt_id .
					"\n";
			}
		} else {
			$logData .=
				"## " .
				t("В процессе сохранения заказа произошла ошибка") .
				":" .
				"\n";
			$logData .= commerce_bifitkassa_check_responce($receipt_id) . "\n";
		}
	} else {
		// log
		if ($receipt_id == 0) {
			$dopText = t("Не удалось сохранить текущий заказ");
		} elseif ($is_update) {
			$dopText = t("Не удалось обновить текущий заказ");
		}

		if ($deliveryType == "SERVICE") {
			$dopText .=
				" " .
				t(
					"по причине того что тип доставки заказа не (Курьер или Самовывоз)!"
				);
		}

		$logData .= $dopText != "" ? "## " . $dopText . "\n" : "";
	}

	// send payment information
	if (
		!$order_deleted &&
		$lk->order_payed &&
		in_array($lk->calc_method, [
			"PREPAY_FULL",
			"PREPAY_PARTIAL",
			"FULL_PAY",
		])
	) {
		// log
		$logData .=
			"## " .
			t("Начинаем процесс передачи данных по оплате заказа...") .
			"\n";

		// address for request
		$url =
			"https://kassa.bifit.com/cashdesk-api/v1/protected/online_orders/payments";

		// headers
		$headers = [
			"Content-Type: application/json",
			"Authorization: Bearer " . $token,
		];

		// data
		$type = "POST";
		$data = "{";
		$data .= '"onlineOrderExternalId":"' . $lk->order_number . '",';
		$data .= '"organizationId":"' . $organization_id . '", ';
		$data .= '"calculationMethod":"' . $lk->calc_method . '", ';
		$data .= '"total": ' . $lk->order_price;
		$data .= "}";

		// log
		$logData .=
			"## " .
			t('Данные по оплате для отправки в базу "БИФИТ Касса"') .
			":" .
			"\n";
		$logData .= $data . "\n";

		// get the ID of the document being processed (id)
		$receipt_id = json_decode(
			commerce_bifitkassa_get_data($url, $headers, $data, $type)
		);

		// log
		if (!is_object($receipt_id) && !is_array($receipt_id)) {
			$logData .=
				"## " .
				t(
					"Передача данных по оплате прошла успешно. Номер документа="
				) .
				$receipt_id .
				"\n";
		} else {
			$logData .=
				"## " .
				t("В процессе передачи данных по оплате произошла ошибка") .
				":" .
				"\n";
			$logData .= commerce_bifitkassa_check_responce($receipt_id) . "\n";
		}
	}

	// save log
	bifitkassa_save_logfile($logData, $lk->order_number);
}

function bifitkassa_organization($type = null, $org_id = null, $orgup = null)
{
	$login = commerce_bifitkassa_check_phone(
		variable_get("commerce_bifitkassa_login", null)
	);
	$password = variable_get("commerce_bifitkassa_password", null);
	$organizations = unserialize(
		variable_get("commerce_bifitkassa_organizations", null)
	);

	if ($login && $password && (!isset($organizations) || isset($orgup))) {
		$tokenData = commerce_bifitkassa_get_token($login, $password);
		$token = $tokenData->access_token;
		$orgData = commerce_bifitkassa_get_organizations($token);
		$organizations = commerce_bifitkassa_create_main_array(
			$orgData,
			$token
		);
		variable_set(
			"commerce_bifitkassa_organizations",
			serialize($organizations)
		);
	}

	$orgsArray = [];
	$tobjArray = [];

	$curr_org = isset($org_id)
		? $org_id
		: variable_get("commerce_bifitkassa_organization", null);

	if (!$curr_org || $curr_org == "") {
		$curr_org = $organizations[0]->id;
	}

	if (isset($organizations)) {
		foreach ($organizations as $key => $org) {
			$orgsArray[$org->id] = $org->name;

			if ($org->id == $curr_org && count($org->traides)) {
				foreach ($org->traides as $k => $trd) {
					$tobjArray[$trd->id] = $trd->name;
				}
			}
		}
	}

	return isset($type) ? $tobjArray : $orgsArray;
}

function bifitkassa_trade_object_update($form, $form_state)
{
	return $form["commerce_bifitkassa_trade_object"];
}

function bifitkassa_organization_update($form, $form_state)
{
	$orgs = bifitkassa_organization(null, null, 1);
	$trds = bifitkassa_organization(1, array_key_first($orgs), null);

	$html =
		'<div class="form-item form-type-select form-item-commerce-bifitkassa-organization">';
	$html .=
		'<label for="edit-commerce-bifitkassa-organization">' .
		t("Организация") .
		"</label>";
	$html .= commerce_bifitkassa_create_selects(
		$orgs,
		"commerce_bifitkassa_organization"
	);
	$html .= "</div>";

	$html .= '<div id="replace_trade_object_div">';
	$html .=
		'<div class="form-item form-type-select form-item-commerce-bifitkassa-trade-object">';
	$html .=
		'<label for="edit-commerce-bifitkassa-trade-object">' .
		t("Торговый объект") .
		"</label>";
	$html .= commerce_bifitkassa_create_selects(
		$trds,
		"commerce_bifitkassa_trade_object"
	);
	$html .= "</div>";
	$html .= "</div>";

	return $html;
}
