<?php
/**
 * @version     5.0.1
 * @package     Jshopping.Plugin
 * @subpackage  bifit
 *
 * @copyright   Copyright (C) 2021—2026 BIFIT. All rights reserved.
 * @license     GNU General Public License version 2 or later
 */

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\String\StringHelper;
use Joomla\Component\Jshopping\Site\Lib\JSFactory;

\defined('_JEXEC') or die();

class PlgJshoppingBifit extends CMSPlugin
{
	public function __construct(&$subject, $config)
	{
		parent::__construct($subject, $config);

		$lang = Factory::getLanguage();
		$lang->load('plg_jshopping_bifit', JPATH_ADMINISTRATOR);
	}

	public static function __createGUID($order)
	{
		// Create GUID (Globally Unique Identifier)
		$guid = '';
		$namespace = rand(11111, 99999);
		$uid = uniqid('', true);
		$data = $namespace;
		$data .= $_SERVER['REQUEST_TIME'];
		$data .= $_SERVER['HTTP_USER_AGENT'];
		$data .= $_SERVER['REMOTE_ADDR'];
		$data .= $_SERVER['REMOTE_PORT'];
		$data .= $order['number'];
		$data .= $order['summ'];
		$hash = StringHelper::strtoupper(
			hash('ripemd128', $uid . $guid . md5($data))
		);
		$guid =
			StringHelper::substr($hash, 0, 8) .
			'-' .
			StringHelper::substr($hash, 8, 4) .
			'-' .
			StringHelper::substr($hash, 12, 4) .
			'-' .
			StringHelper::substr($hash, 16, 4) .
			'-' .
			StringHelper::substr($hash, 20, 12);

		return StringHelper::strtolower($order['cms'] . '-' . $guid);
	}

	public static function __formatprc($price)
	{
		// форматируем цену
		$price = number_format($price, 2, '.', '');

		return $price;
	}

	public static function __checkInn($inn, $type = 1)
	{
		// проверяем заполненность
		if (isset($inn) && StringHelper::trim($inn) != '') {
			if ($type == 2) {
				// только цифры в количестве 10 шт.
				$inn = !preg_match("/^\d{10}$/", StringHelper::trim($inn))
					? Text::_('PLG_JSHOPPING_BIFIT_WRONG_INN')
					: StringHelper::trim($inn);
			} elseif ($type == 1) {
				// только цифры в количестве 12 шт.
				$inn = !preg_match("/^\d{12}$/", StringHelper::trim($inn))
					? Text::_('PLG_JSHOPPING_BIFIT_WRONG_INN')
					: StringHelper::trim($inn);
			} else {
				// только цифры в количестве 10 или 12 шт.
				$inn = !preg_match(
					"/^\d{10}$|^\d{12}$/",
					StringHelper::trim($inn)
				)
					? Text::_('PLG_JSHOPPING_BIFIT_WRONG_INN')
					: StringHelper::trim($inn);
			}
		} else {
			// ИНН пуст или не найден!
			$inn = Text::_('PLG_JSHOPPING_BIFIT_EMPTY_INN');
		}

		return $inn;
	}

	public static function __checkPhone($tel)
	{
		// проверяем заполненность
		if (isset($tel) && StringHelper::trim($tel) != '') {
			// удаляем лишние символы
			$phone = StringHelper::str_ireplace(
				['+', '(', ')', '-', ' '],
				'',
				StringHelper::trim($tel)
			);

			// получаем первую цифру
			$first = StringHelper::substr($phone, "0", 1);

			// если номер из 10 цифр и первая цифра не 7, то добавляем 7
			$phone =
				StringHelper::strlen($phone) == 10 && $first != 7
					? '7' . $phone
					: $phone;

			// окончательная проверка
			$phone = !preg_match("/^[0-9]{11}+$/", $phone)
				? Text::_('PLG_JSHOPPING_BIFIT_WRONG_PHONE')
				: $phone;
		} else {
			// Телефон пуст или не найден!
			$phone = Text::_('PLG_JSHOPPING_BIFIT_EMPTY_PHONE');
		}

		return $phone;
	}

	public static function __checkEmail($email)
	{
		// проверка модели записи
		$email = !preg_match("/^\+\d+$|^\S+@\S+$/", $email)
			? Text::_('PLG_JSHOPPING_BIFIT_WRONG_EMAIL')
			: $email;

		return $email;
	}

	public static function __checkProductPrice(
		$item_price,
		$order_subtotal,
		$order_discount = 0,
		$order_payment = 0
	) {
		// если скидка больше нуля
		if ($order_discount > 0) {
			// вычисляем процент скидки
			$percent = ($order_discount * 100) / $order_subtotal;

			// уменьшаем цену товара согласно скидки
			$item_price = $item_price - $item_price * ($percent / 100);
		}

		// если система оплаты больше нуля
		if ($order_payment > 0) {
			// вычисляем процент наценки
			$percent =
				($order_payment * 100) / ($order_subtotal + $order_payment);

			// увеличиваем цену товара согласно наценки
			$item_price = $item_price + $item_price * ($percent / 100);
		}

		return $item_price;
	}

	public static function __checkResponce($response)
	{
		$mess = '';

		if (
			is_object($response) &&
			($response->type == 'ERROR' || $response->status == 400)
		) {
			$mess = '';
			if ($response->status == 400) {
				$mess .= $response->error . "\n";
				$mess .= $response->message . "\n";
			} else {
				$array = json_decode(json_encode($response), true);
				array_walk_recursive($array, function ($item, $key) use (
					&$mess
				) {
					if ($item != 'ERROR' && StringHelper::trim($item != '')) {
						$mess .= $key . ': ' . $item . "\n";
					}
				});
			}
		} elseif (is_object($response)) {
			$array = json_decode(json_encode($response), true);
			array_walk_recursive($array, function ($item, $key) use (&$mess) {
				if (StringHelper::trim($item != '')) {
					$mess .= $key . ': ' . $item . "\n";
				}
			});
		}

		return $mess;
	}

	public static function __loadData(
		$url,
		$headers = false,
		$data = false,
		$type = 'POST'
	) {
		// инициализируем сеанс cURL
		$ch = curl_init($url);

		// назначаем тип запроса
		if ($type != 'POST') {
			curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $type);
		} else {
			curl_setopt($ch, CURLOPT_POST, true);
		}

		// добавляем данные
		if ($data) {
			curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
		}

		// возврат результата в качестве строки
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_ENCODING, '');
		curl_setopt($ch, CURLOPT_MAXREDIRS, 10);
		curl_setopt($ch, CURLOPT_TIMEOUT, 0);
		curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
		curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);

		// добавляем headers
		if ($headers) {
			curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
		}

		// получаем ответ
		$output = curl_exec($ch);

		// завершаем сеанс cURL
		curl_close($ch);

		return $output;
	}

	public static function __saveReport(
		$test,
		$bifit,
		$receipt,
		$receipt_id,
		$doc,
		$object,
		$idemp_key
	) {
		$app = Factory::getApplication();
		$db = Factory::getDBO();

		// выводим свойство операции
		$log =
			$test != ''
				? '####### ' .
					Text::_('PLG_JSHOPPING_BIFIT_REQUEST_TEST') .
					"\n"
				: '####### ' .
					Text::_('PLG_JSHOPPING_BIFIT_REQUEST_BATTLE') .
					"\n";

		// устанавливаем часовой пояс согласно настройкам сайта
		date_default_timezone_set($app->getCfg('offset'));
		$log .= '####### ' . date('Y-m-d H:i:s', time()) . "\n";

		// ОТВЕТ #1

		$log .=
			'####### ' . Text::_('PLG_JSHOPPING_BIFIT_ANSWER') . ' #1: ' . "\n";

		// данные ответа #1
		$token_error = null;

		if (is_object($bifit)) {
			foreach ($bifit as $key => $val) {
				if ($key == 'error') {
					$token_error = 1;
				}
				$log .= $key . ': ' . $val . "\n";
			}
		}

		// ЧЕК

		// сформированный чек для отправки
		$log .=
			'####### ' .
			Text::_('PLG_JSHOPPING_BIFIT_CHECK') .
			': ' .
			"\n" .
			$receipt .
			"\n";

		if (!$token_error) {
			// КЛЮЧ ИДЕМПОТЕНТНОСТИ

			$log .=
				'####### ' .
				Text::_('PLG_JSHOPPING_BIFIT_IDEMPOTENCY_KEY') .
				': ' .
				"\n" .
				$idemp_key .
				"\n";

			// ОТВЕТ #2

			$log .=
				'####### ' .
				Text::_('PLG_JSHOPPING_BIFIT_ANSWER') .
				' #2: ' .
				"\n";

			// данные ответа #2
			if (is_object($receipt_id)) {
				$log .= self::__checkResponce($receipt_id);
			} else {
				$log .=
					Text::_('PLG_JSHOPPING_BIFIT_DOCUMENT_NUMBER') .
					': ' .
					$receipt_id .
					"\n";

				// ОТВЕТ #3

				$log .=
					'####### ' .
					Text::_('PLG_JSHOPPING_BIFIT_ANSWER') .
					' #3: ' .
					"\n";

				if (is_object($doc)) {
					$log .= self::__checkResponce($doc);
				}

				// ССЫЛКА НА ЧЕК

				// сформированная ссылка на чек
				$log .=
					'####### ' .
					Text::_('PLG_JSHOPPING_BIFIT_LINK_TO_CHECK') .
					': ' .
					"\n" .
					$object->receipt_link;
			}
		}

		$object->log = $log;

		// СОХРАНЯЕМ В БД
		$db->insertObject(
			$db->getPrefix() . 'jshopping_bifit_reports',
			$object
		);
	}

	public function __mainProcess($order, $test, $token, $i)
	{
		// 1. АВТОРИЗАЦИЯ
		//    процесс необходимый для подключения клиента к фискальному процессингу

		// адрес для POST-запроса
		$url = 'https://fp' . $test . '.bifit.com/processing-api/oauth/token';

		// данные ссылки
		$data = 'token=' . $token;
		$data .= '&client_id=processing-connector-token';
		$data .= '&client_secret=processing-connector-token';
		$data .= '&grant_type=token';

		// headers
		$headers = ['Content-Type: application/x-www-form-urlencoded'];

		// декодируем полученный json-ответ
		$bifit = json_decode(self::__loadData($url, $headers, $data));

		// 2. ФОРМИРОВАНИЕ КОНТЕНТА ЧЕКА
		//    формирование информации для дальнейшей отправки на фискализацию

		// параметры плагина
		/// имя кассира
		$cashier_name = $this->params->get('cashier_name', 'Иванов И.И.');
		/// ИНН кассира
		$cashier_inn = $this->params->get('cashier_inn', '123456789012');
		/// тип документа
		$arr_rec_type = [
			'1' => 'SALE',
			'2' => 'SALE_RETURN',
			'3' => 'PURCHASE',
			'4' => 'PURCHASE_RETURN'
		];
		$receipt_type = $arr_rec_type[$i];
		/// система налогов
		$tax_system = $this->params->get(
			'tax_system_' . $i,
			'SIMPLIFIED_WITH_EXPENSE'
		);
		/// признак способа расчёта
		$calc_method = $this->params->get('calculation_method', 'FULL_PAY');
		/// единый НДС
		$vat_mode = $this->params->get('vat_mode', 1);
		/// НДС
		$vat = $this->params->get('vat', 'WITHOUT_VAT');
		/// тип оплаты
		$payment_type = $this->params->get('payment_type', 'CARD');
		/// место расчётов
		$payment_address = $this->params->get(
			'payment_address',
			'https://site.ru'
		);
		/// тип адреса клиента
		$address_type = $this->params->get('address_type', 'email');
		/// ФИО клиента
		$fio = $this->params->get('fio', 0);
		/// адрес клиента в чеке (телефон или эл.почта)
		if ($address_type == 'phone' && $order->phone) {
			$address = self::__checkPhone($order->phone);
		} elseif ($address_type == 'phone' && $order->mobil_phone) {
			$address = self::__checkPhone($order->mobil_phone);
		} else {
			$address = self::__checkEmail($order->email);
		}

		// позиции оплаты заказа
		/// стоимость способа оплаты
		$o_payment = $order->order_payment;
		/// стоимость упаковки
		$o_package = $order->order_package;
		/// стоимость доставки
		$o_shipping = $order->order_shipping;

		// вывод цен с НДС или без НДС ? с НДС : без НДС;
		$tax_out =
			JSFactory::getConfig()->display_price_front_current == 1 ? 1 : null;

		// если цены на фронте без НДС
		if ($tax_out) {
			// корректируем цену способа оплаты с учётом НДС
			if ($o_payment > 0) {
				$paytaxlist = Factory::getSession()->get(
					"jshop_price_payment_tax_list",
					[]
				);
				$o_payment =
					count($paytaxlist) &&
					$paytaxlist[array_key_first($paytaxlist)] > 0
						? $order->order_payment +
							$paytaxlist[array_key_first($paytaxlist)]
						: $order->order_payment;
			}

			// корректируем цену упаковки с учётом НДС
			if ($o_package > 0) {
				$pactaxlist = Factory::getSession()->get(
					"jshop_price_package_tax_list",
					[]
				);
				$pactax = count($pactaxlist) ? array_key_first($pactaxlist) : 0;
				$o_package =
					$tax_out &&
					count($pactaxlist) &&
					$pactaxlist[array_key_first($pactaxlist)] > 0
						? $order->order_package +
							$pactaxlist[array_key_first($pactaxlist)]
						: $order->order_package;
			}

			// корректируем цену способа доставки с учётом НДС
			if ($o_shipping > 0) {
				$shptaxlist = Factory::getSession()->get(
					"jshop_price_shipping_tax_list",
					[]
				);
				$shptax = count($shptaxlist) ? array_key_first($shptaxlist) : 0;
				$o_shipping =
					$tax_out &&
					count($shptaxlist) &&
					$shptaxlist[array_key_first($shptaxlist)] > 0
						? $order->order_shipping +
							$shptaxlist[array_key_first($shptaxlist)]
						: $order->order_shipping;
			}
		}

		// параметры массива
		/// промежуточный итог
		$o_subtotal = 0;
		/// массив товаров для заполнения
		$r_products = [];
		/// массив товаров заказа
		$products = $order->getAllItems();

		// перебираем массив товаров заказа
		foreach ($products as $key_id => $product) {
			/// корректируем цену товара с учётом НДС
			$prod_price =
				$tax_out && $product->product_tax > 0
					? self::__formatprc(
						$product->product_item_price +
							($product->product_item_price / 100) *
								$product->product_tax
					)
					: self::__formatprc($product->product_item_price);

			/// корректируем цену товара с учётом скидок
			$p_price = self::__formatprc(
				self::__checkProductPrice(
					$prod_price,
					$order->order_subtotal,
					$order->order_discount,
					$order->order_payment
				)
			);
			/// формируем промежуточный итог
			$o_subtotal += $p_price * $product->product_quantity;

			/// очищаем имя товара от спец-символов
			$p_name = preg_replace(
				'/[^\p{L}0-9\-]/iu',
				' ',
				html_entity_decode($product->product_name)
			);
			$p_name = preg_replace('/ +/', ' ', StringHelper::trim($p_name));

			/// формируем массив товаров для чека
			$r_products[$key_id] = new stdClass();
			$r_products[$key_id]->product_name = $p_name;
			$r_products[$key_id]->product_item_price = $p_price;
			$r_products[$key_id]->product_quantity = $product->product_quantity;
			$r_products[$key_id]->product_tax = $product->product_tax;
		}

		// сверка итоговых сумм заказа и чека
		$total_1 = self::__formatprc($o_subtotal + $o_package + $o_shipping);
		$total_2 = self::__formatprc($order->order_total);
		$diff =
			$total_1 > 0 && $total_2 > 0 && $total_1 != $total_2
				? abs($total_1 - $total_2)
				: 0;

		// параметры для чека
		$items = '';

		// формируем данные чека
		foreach ($r_products as $key_id => $product) {
			/// цена товара
			$p_price = $product->product_item_price;

			/// корректировка цены товара согласно сверки
			if ($diff > 0 && $p_price > $diff) {
				$p_price =
					$total_2 > $total_1
						? $p_price + $diff / $product->product_quantity
						: $p_price - $diff / $product->product_quantity;
				$diff = 0;
			}

			/// выводим данные товара в чек
			$items .= $items != '' ? ', ' : '';
			$items .= '{';
			$items .= '"calculationMethod":"' . $calc_method . '", ';
			$items .= '"paymentSubject":"PRODUCT", ';
			$items .= '"name":"' . $product->product_name . '", ';
			$items .= '"price":' . self::__formatprc($p_price) . ', ';
			$items .= '"quantity":' . $product->product_quantity . ', ';
			$items .=
				$vat_mode == 1
					? '"vat":"' . $vat . '", '
					: '"vat":"VAT_' . floor($product->product_tax) . '", ';
			$items .=
				'"total":' .
				self::__formatprc($p_price * $product->product_quantity);
			$items .= '}';
		}

		// упаковка
		if ($o_package > 0) {
			// выводим данные упаковки в чек
			$items .= $items != '' ? ', ' : '';
			$items .= '{';
			$items .= '"calculationMethod":"' . $calc_method . '", ';
			$items .= '"paymentSubject":"PRODUCT", ';
			$items .= '"name":"Упаковка", ';
			$items .= '"price":' . self::__formatprc($o_package) . ', ';
			$items .= '"quantity":1, ';
			$items .=
				$vat_mode == 1
					? '"vat":"' . $vat . '", '
					: '"vat":"VAT_' . floor($pactax) . '", ';
			$items .= '"total":' . self::__formatprc($o_package);
			$items .= '}';
		}

		// доставка
		if ($o_shipping > 0) {
			/// выводим данные способа доставки в чек
			$items .= $items != '' ? ', ' : '';
			$items .= '{';
			$items .= '"calculationMethod":"' . $calc_method . '", ';
			$items .= '"paymentSubject":"SERVICE", ';
			$items .= '"name":"Доставка", ';
			$items .= '"price":' . self::__formatprc($o_shipping) . ', ';
			$items .= '"quantity":1, ';
			$items .=
				$vat_mode == 1
					? '"vat":"' . $vat . '", '
					: '"vat":"VAT_' . floor($shptax) . '", ';
			$items .= '"total":' . self::__formatprc($o_shipping);
			$items .= '}';
		}

		// формируем чек
		$receipt = '{';
		$receipt .=
			'"type":"' .
			$receipt_type .
			'", "taxSystem":"' .
			$tax_system .
			'", ';
		$receipt .= '"cashier":{';
		$receipt .= '"name":"' . $cashier_name . '", ';
		$receipt .= '"inn":"' . self::__checkInn($cashier_inn, 1) . '"';
		$receipt .= '}, ';
		$receipt .= '"client":{';
		$receipt .= $fio
			? '"name":"' .
				StringHelper::trim(
					$order->l_name . ' ' . $order->f_name . ' ' . $order->m_name
				) .
				'", '
			: '';
		$receipt .= '"address":"' . $address . '"';
		$receipt .= '}, ';
		$receipt .= '"paymentAddress":"' . $payment_address . '", ';
		$receipt .= '"items":[' . $items . '], ';
		$receipt .= '"total":' . self::__formatprc($order->order_total) . ', ';
		$receipt .=
			'"payments":{"' .
			$payment_type .
			'": ' .
			self::__formatprc($order->order_total) .
			'}';
		$receipt .= '}';

		// 3. ОТПРАВКА ЧЕКА В ФИСКАЛЬНЫЙ ПРОЦЕССИНГ
		//    передача контента чека в фискальный процессинг для дальнейшей фискализации

		// адрес для POST-запроса
		$url =
			'https://fp' .
			$test .
			'.bifit.com/processing-api/protected/documents/registration/receipts';

		// данные чека
		$data = $receipt;

		// ключ идемпотентности (Idempotency-Key)
		$o = [];
		$o['number'] = $order->order_number;
		$o['summ'] = self::__formatprc($order->order_total);
		$o['cms'] = 'Joomshopping';
		$idemp_key = self::__createGUID($o);

		// headers
		$headers = [
			'Content-Type: application/json',
			'Authorization: Bearer ' . $bifit->access_token,
			'Idempotency-Key: ' . $idemp_key
		];

		// получаем идентификатор документа на процессинге (id)
		$receipt_id = json_decode(self::__loadData($url, $headers, $data));

		// 4. ПОЛУЧЕНИЕ ДОКУМЕНТА ПО ID
		//    получение документа с фискальными признаками по его номеру

		// параметры
		$doc = '';

		if (!is_object($receipt_id)) {
			// ссылка для GET-запроса
			$url =
				'https://fp' .
				$test .
				'.bifit.com/processing-api/protected/documents/' .
				$receipt_id;

			// headers
			$headers = ['Authorization: Bearer ' . $bifit->access_token];

			// получаем документ
			$doc = json_decode(self::__loadData($url, $headers, false, 'GET'));
		}

		// ССЫЛКА НА ЧЕК

		// формируем ссылку на чек
		$receipt_link = !is_object($receipt_id)
			? 'https://fp' .
				$test .
				'.bifit.com/processing-api/receipts/' .
				$idemp_key
			: 'XXX';

		// ИСТОРИЯ ОПЕРАЦИИ

		// параметры объекта
		$object = new stdClass();
		/// id заказа
		$object->order_id = $order->order_id;
		/// номер заказа
		$object->order_number = $order->order_number;
		/// дата заказа
		$object->order_date = $order->order_date;
		/// дата операции
		$object->action_date = date('Y-m-d H:i:s');
		/// ссылка на чек
		$object->receipt_link = $receipt_link;
		/// id полученного документа
		$object->receipt_id = !is_object($receipt_id) ? $receipt_id : '000';
		/// тип проводимого документа
		$object->receipt_type = $receipt_type;
		/// лог операции (будет заполнен далее)
		$object->log = '';

		// сохраняем операцию в историю
		self::__saveReport(
			$test,
			$bifit,
			$receipt,
			$receipt_id,
			$doc,
			$object,
			$idemp_key
		);
	}

	public function onBeforeDisplayCheckoutFinish(&$text, &$order_id)
	{
		// получаем заказ
		$order = JSFactory::getTable('order');
		$order->load($order_id);

		// данные заказа
		$status_id = $order->order_status;
		$payment_id = $order->payment_method_id;

		// параметры плагина
		/// выбранный режим работы
		$test = $this->params->get('test', 1) ? '-test' : '';
		/// токен для соединения
		$token =
			$test != ''
				? 'P5cKbUUD9uSSrSlGdzspLblvBnD0GzTAE0cLmAPSEMxJ79DtLE'
				: $this->params->get('token', null);

		// перебираем вкладки типов документа
		for ($i = 1; $i <= 4; $i++) {
			/// узнаём нужно ли использовать данный тип документа
			$use_type = $this->params->get('use_type_' . $i, null);
			/// если использовать нужно
			if ($use_type) {
				//// выбранный способ(ы) оплаты
				$pm_ids = $this->params->get('pm_ids_' . $i, null);
				//// выбранный статус(ы) заказа
				$pm_statuses = $this->params->get('pm_statuses_' . $i, null);
				//// если совпадает способ оплаты и статус заказа
				if (
					isset($pm_ids) &&
					isset($pm_statuses) &&
					in_array($payment_id, $pm_ids) &&
					in_array($status_id, $pm_statuses) &&
					$token != ''
				) {
					///// запускаем БИФИТ-процесс
					self::__mainProcess($order, $test, $token, $i);
				}
			}
		}
	}

	public function onAfterChangeOrderStatusAdmin(
		&$order_id,
		&$status,
		&$status_id,
		&$notify,
		&$comments,
		&$include_comment,
		&$view_order,
		&$prev_order_status
	) {
		// получаем заказ
		$order = JSFactory::getTable('order');
		$order->load($order_id);

		// данные заказа
		$status_id = $status;
		$payment_id = $order->payment_method_id;

		// параметры плагина
		/// выбранный режим работы
		$test = $this->params->get('test', 1) ? '-test' : '';
		/// токен для соединения
		$token =
			$test != ''
				? 'P5cKbUUD9uSSrSlGdzspLblvBnD0GzTAE0cLmAPSEMxJ79DtLE'
				: $this->params->get('token', null);

		// перебираем вкладки типов документа
		for ($i = 1; $i <= 4; $i++) {
			/// узнаём нужно ли использовать данный тип документа
			$use_type = $this->params->get('use_type_' . $i, null);
			/// если использовать нужно
			if ($use_type) {
				//// выбранный способ(ы) оплаты
				$pm_ids = $this->params->get('pm_ids_' . $i, null);
				//// выбранный статус(ы) заказа
				$pm_statuses = $this->params->get('pm_statuses_' . $i, null);
				//// если совпадает способ оплаты и статус заказа
				if (
					isset($pm_ids) &&
					isset($pm_statuses) &&
					in_array($payment_id, $pm_ids) &&
					in_array($status_id, $pm_statuses) &&
					$token != ''
				) {
					///// запускаем БИФИТ-процесс
					self::__mainProcess($order, $test, $token, $i);
				}
			}
		}
	}

	public function onBeforeShowOrder(&$view)
	{
		$order = $view->order;
		$order_id = $order->order_id;

		$db = Factory::getDbo();
		$query = $db->getQuery(true);
		$query
			->clear()
			->select('*')
			->from($db->quoteName($db->getPrefix() . 'jshopping_bifit_reports'))
			->where($db->quoteName('order_id') . ' = ' . $db->quote($order_id));
		$db->setQuery($query);
		$reports = $db->loadObjectList();
		$db->execute();
		$btns = '';

		if (
			isset($reports) &&
			(is_object($reports) || is_array($reports)) &&
			count($reports)
		) {
			foreach ($reports as $k => $action) {
				$btns .=
					'<a target="_blank" href="' .
					$action->receipt_link .
					'" class="btn btn-sm btn-primary me-1" title="' .
					Text::_('Ссылка на чек') .
					'">' .
					Text::_(
						'PLG_JSHOPPING_BIFIT_RECEIPT_TYPE_' .
							$action->receipt_type
					) .
					'</a>';
			}
		}

		if ($btns != '') {
			$html = '<tr>';
			$html .=
				'<td>' . Text::_('PLG_JSHOPPING_BIFIT_RECEIPTS') . ':</td>';
			$html .= '<td colspan="2">' . $btns . '</td>';
			$html .= '</tr>';

			$view->_update_status_html .= $html;
		}
	}
}
