<?php

class shopBifitkassaPlugin extends shopPlugin
{
	const PLUGIN_ID = 'bifitkassa';

	public static function __createGUID($order)
	{
		// Create GUID (Globally Unique Identifier)
		$guid = '';
		$namespace = rand(11111, 99999);
		$uid = uniqid('', true);
		$data = $namespace;
		$data .= $_SERVER['REQUEST_TIME'];
		$data .= $_SERVER['HTTP_USER_AGENT'];
		$data .= $_SERVER['REMOTE_ADDR'];
		$data .= $_SERVER['REMOTE_PORT'];
		$data .= $order['number'];
		$data .= $order['summ'];
		$hash = strtoupper(hash('ripemd128', $uid . $guid . md5($data)));
		$guid =
			substr($hash, 0, 8) .
			'-' .
			substr($hash, 8, 4) .
			'-' .
			substr($hash, 12, 4) .
			'-' .
			substr($hash, 16, 4) .
			'-' .
			substr($hash, 20, 12);

		return strtolower($order['cms'] . '-' . $guid);
	}

	public static function __clearString($str)
	{
		$arr = ["\\", "/", "\n", "\r", "\t", "\x08", "\x0c"];
		$str = str_replace($arr, " ", $str);
		$str = str_replace("Ð", "D", $str);
		$str = preg_replace(
			"/\s\s+/",
			" ",
			preg_replace(
				"/[^ а-яА-ЯёЁ0-9a-zA-Z?@№#()!,-.:;]/ui",
				"",
				strip_tags(str_replace(" ", " ", $str))
			)
		);

		return trim($str);
	}

	static function __getOrderTransIds($order_id, $org_id, $token)
	{
		// link
		$url =
			"https://kassa.bifit.com/cashdesk-api/v1/protected/online_orders/details/" .
			$order_id .
			"?organization_id=" .
			$org_id;

		// headers
		$headers = [
			"Content-Type: application/x-www-form-urlencoded",
			"Authorization: Bearer " . $token
		];

		// data
		$type = "GET";
		$data = false;

		// result
		$orderContainer = json_decode(
			self::__loadData($url, $headers, $data, $type)
		);
		$ids = [];

		if (is_object($orderContainer)) {
			$onlineOrderPayments = $orderContainer->onlineOrderPayments;

			if (count($onlineOrderPayments)) {
				foreach ($onlineOrderPayments as $onlinePayment) {
					$ids[$onlinePayment->id] = $onlinePayment->transactionId;
				}
			}
		}

		return $ids;
	}

	static function __formatprc($price)
	{
		$dec_count = 2;
		$price =
			$dec_count < 0
				? round($price, $dec_count)
				: number_format($price, $dec_count, '.', '');

		return $price;
	}

	static function __checkInn($inn, $type = 1)
	{
		if (isset($inn) && trim($inn) != '') {
			if ($type == 2) {
				$inn = !preg_match("/^\d{10}$/", trim($inn))
					? _wp('ИНН задан в неверном формате!')
					: trim($inn);
			} elseif ($type == 1) {
				$inn = !preg_match("/^\d{12}$/", trim($inn))
					? _wp('ИНН задан в неверном формате!')
					: trim($inn);
			} else {
				$inn = !preg_match("/^\d{10}$|^\d{12}$/", trim($inn))
					? _wp('ИНН задан в неверном формате!')
					: trim($inn);
			}
		} else {
			$inn = _wp('ИНН пуст или не найден!');
		}

		return $inn;
	}

	static function __checkPhone($tel)
	{
		if (isset($tel) && trim($tel) != '') {
			$phone = str_replace(
				array('+', '(', ')', '-', ' '),
				'',
				trim($tel)
			);
			$first = substr($phone, "0", 1);
			$phone =
				strlen($phone) == 10 && $first != 7 ? '7' . $phone : $phone;
			$phone = !preg_match("/^[0-9]{11}+$/", $phone)
				? _wp('Телефон задан в неверном формате!')
				: $phone;
		} else {
			$phone = _wp('Телефон пуст или не найден!');
		}

		return $phone;
	}

	static function __checkEmail($email)
	{
		$email = !preg_match("/^\+\d+$|^\S+@\S+$/", $email)
			? _wp('E-mail задан в неверном формате!')
			: $email;

		return $email;
	}

	static function __checkProductPrice(
		$item_price,
		$order_subtotal,
		$order_discount = 0,
		$order_payment = 0
	) {
		if ($order_discount > 0) {
			$percent = ($order_discount * 100) / $order_subtotal;
			$item_price = $item_price - $item_price * ($percent / 100);
		}

		if ($order_payment > 0) {
			$percent =
				($order_payment * 100) / ($order_subtotal - $order_discount);

			$item_price = $item_price + $item_price * ($percent / 100);
		}

		return $item_price;
	}

	static function __checkResponce($response)
	{
		$mess = '';

		if (
			is_object($response) &&
			($response->type == 'ERROR' || $response->status == 400)
		) {
			$mess = '';
			if ($response->status == 400) {
				$mess .= $response->error . "\n";
				$mess .= $response->message . "\n";
			} else {
				$array = json_decode(json_encode($response), true);
				array_walk_recursive($array, function ($item, $key) use (
					&$mess
				) {
					if ($item != 'ERROR' && trim($item != '')) {
						$mess .= $key . ': ' . $item . "\n";
					}
				});
			}
		} elseif (is_object($response)) {
			$array = json_decode(json_encode($response), true);
			array_walk_recursive($array, function ($item, $key) use (&$mess) {
				if (trim($item != '')) {
					$mess .= $key . ': ' . $item . "\n";
				}
			});
		}

		return $mess;
	}

	static function __loadData(
		$url,
		$headers = false,
		$data = false,
		$type = 'POST'
	) {
		$ch = curl_init($url);

		if ($type != 'POST') {
			curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $type);
		} else {
			curl_setopt($ch, CURLOPT_POST, true);
		}

		if ($data) {
			curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
		}

		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_ENCODING, '');
		curl_setopt($ch, CURLOPT_MAXREDIRS, 10);
		curl_setopt($ch, CURLOPT_TIMEOUT, 0);
		curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
		curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);

		if ($headers) {
			curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
		}

		$output = curl_exec($ch);
		curl_close($ch);

		return $output;
	}

	static function __getToken($login, $password)
	{
		$signature = urlencode(base64_encode(hash('sha256', $password, true)));
		$data = 'username=' . $login;
		$data .= '&password=' . $signature;
		$data .= '&client_id=cashdesk-rest-client';
		$data .= '&client_secret=cashdesk-rest-client';
		$data .= '&grant_type=password';
		$url = 'https://kassa.bifit.com/cashdesk-api/v1/oauth/token';
		$headers = array('Content-Type: application/x-www-form-urlencoded');
		$tokenData = json_decode(self::__loadData($url, $headers, $data));

		return $tokenData;
	}

	static function __getOrganizations($token)
	{
		$url =
			'https://kassa.bifit.com/cashdesk-api/v1/protected/organizations/list/read';
		$headers = array('Authorization: Bearer ' . $token);
		$orgData = json_decode(self::__loadData($url, $headers));

		return $orgData;
	}

	static function __getTradeObjects($org_id, $token)
	{
		$data = 'organization_id=' . $org_id;
		$url =
			'https://kassa.bifit.com/cashdesk-api/v1/protected/trade_objects/list/read';
		$headers = array(
			'Content-Type: application/x-www-form-urlencoded',
			'Authorization: Bearer ' . $token
		);
		$tradeData = json_decode(self::__loadData($url, $headers, $data));

		return $tradeData;
	}

	static function __createMainArray($orgData, $token)
	{
		$arr = [];

		foreach ($orgData as $key => $org) {
			$arr[$key] = new stdClass();
			$arr[$key]->id = $org->id;
			$arr[$key]->name = $org->name;
			$arr[$key]->traides = [];

			$tradeData = self::__getTradeObjects($org->id, $token);

			foreach ($tradeData as $k => $trd) {
				$arr[$key]->traides[$k] = new stdClass();
				$arr[$key]->traides[$k]->id = $trd->id;
				$arr[$key]->traides[$k]->name = $trd->name;
			}
		}

		return $arr;
	}

	static function __createSelects($name)
	{
		$plugin_id = self::PLUGIN_ID;
		$plugin = waSystem::getInstance()->getPlugin($plugin_id, true);
		$organizations =
			is_array($plugin->getSettings('organizations')) &&
			array_key_first($plugin->getSettings('organizations')) === 0
				? $plugin->getSettings('organizations')
				: [];
		$data = [];

		if (is_array($organizations) && count($organizations)) {
			$org_id =
				$plugin->getSettings('organization') != -1 &&
				$plugin->getSettings('organization') != 1
					? $plugin->getSettings('organization')
					: $organizations[0]['id'];

			foreach ($organizations as $key => $org) {
				if ($name == 'organization') {
					$data[] = [
						'value' => $org['id'],
						'title' => $org['name']
					];
				}

				if ($name == 'trade_object' && $org['id'] != $org_id) {
					continue;
				}

				if ($name != 'organization') {
					foreach ($org['traides'] as $k => $trd) {
						$data[] = [
							'value' => $trd['id'],
							'title' => $trd['name']
						];
					}
				}
			}
		}

		return $data;
	}

	static function __writeLog($data, $order_number)
	{
		$file =
			dirname(__FILE__) .
			'/../../../../../wa-log/shop/plugins/bifitkassa/log_' .
			$order_number .
			'.txt';
		$content = file_exists($file) ? file_get_contents($file) : '';
		$content .=
			is_array($data) || is_object($data)
				? print_r($data, 1) . "\n"
				: $data . "\n";

		file_put_contents($file, $content);
	}

	static function __saveReport($order, $receipt_id, $doc, $bifit)
	{
		// display the property of the operation
		
		$log  = '######################' . "\n";
		$log .= '## ФИСКАЛИЗАЦИЯ ######' . "\n";
		$log .= '######################' . "\n";
		// set the time zone according to the site settings
		$log .= '## ' . date('Y-m-d H:i:s', time()) . "\n";
		$log .= '######################' . "\n";
		$log .=
			$order->test != ''
				? '## ' . _wp('Тестовый запрос') . "\n"
				: '## ' . _wp('Боевой запрос') . "\n";

		// ANSWER #1

		$log .= '####### ' . _wp('Ответ') . ' #1: ' . "\n";

		// response data # 1
		$token_error = null;

		if (is_object($bifit)) {
			foreach ($bifit as $key => $val) {
				if ($key == 'error') {
					$token_error = 1;
				}

				$log .= $key . ': ' . $val . "\n";
			}
		}

		// RECEIPT

		// generated receipt for sending
		$log .= '####### ' . _wp('Чек') . ': ' . "\n" . $order->receipt . "\n";

		if (!$token_error) {
			// IDEMPOTENCY KEY

			$log .=
				'####### ' .
				_wp('Ключ идемпотентности') .
				': ' .
				"\n" .
				$order->idemp_key .
				"\n";

			// ANSWER #2

			$log .= '####### ' . _wp('Ответ') . ' #2: ' . "\n";

			// response data #2
			if (is_object($receipt_id)) {
				$log .= self::__checkResponce($receipt_id);
				$order->receipt_id = _wp('нет');
			} else {
				$order->receipt_id = $receipt_id;
				$log .= 'Номер документа: ' . $receipt_id . "\n";

				// ANSWER #3

				$log .= '####### ' . _wp('Ответ') . ' #3: ' . "\n";

				if (is_object($doc)) {
					$log .= self::__checkResponce($doc);
				}

				// LINK TO THE CHECK

				// generated link to the check
				$log .=
					'####### ' .
					_wp('Ссылка на чек') .
					': ' .
					"\n" .
					$order->receipt_link;
			}
		}

		$order->log = $log;

		// SAVE

		self::__writeLog($log, $order->number);

		// object params
		$log_model = new shopOrderLogModel();
		$log_model->add([
			'order_id' => $order->id,
			'action_id' => '',
			'contact_id' => null,
			'before_state_id' => "new",
			'after_state_id' => "new",
			'text' => _wp('Ссылка на чек') . ': ' . "\n" . $order->receipt_link
		]);
	}

	/**
	 * @param $order_id
	 * @return shopOrder|false
	 */
	private function getOrder($order_id)
	{
		try {
			$order = new shopOrder($order_id);
		} catch (Exception $e) {
			waLog::log('Order not found: ' . $order_id, self::LOG_FILE);

			return false;
		}

		return $order;
	}

	static function addressTypeValues()
	{
		$res = array(
			array(
				'value' => 'phone',
				'title' => 'Телефон'
			),
			array(
				'value' => 'email',
				'title' => 'E-mail'
			)
		);

		return $res;
	}

	static function receiptTypeValues()
	{
		$res = array(
			array(
				'value' => 'SALE',
				'title' => 'Приход'
			),
			array(
				'value' => 'SALE_RETURN',
				'title' => 'Возврат прихода'
			),
			array(
				'value' => 'PURCHASE',
				'title' => 'Расход'
			),
			array(
				'value' => 'PURCHASE_RETURN',
				'title' => 'Возврат расхода'
			)
		);

		return $res;
	}

	static function taxSystemValues()
	{
		$res = array(
			array(
				'value' => 'COMMON',
				'title' => 'ОСН'
			),
			array(
				'value' => 'SIMPLIFIED',
				'title' => 'УСН доход'
			),
			array(
				'value' => 'SIMPLIFIED_WITH_EXPENSE',
				'title' => 'УСН доход-расход'
			),
			array(
				'value' => 'ENVD',
				'title' => 'ЕНВД'
			),
			array(
				'value' => 'COMMON_AGRICULTURAL',
				'title' => 'ЕСХН'
			),
			array(
				'value' => 'PATENT',
				'title' => 'ПАТЕНТ'
			)
		);

		return $res;
	}

	static function calculationMethodValues()
	{
		$res = array(
			array(
				'value' => 'PREPAY_FULL',
				'title' =>
					'Полная предварительная оплата до момента передачи предмета расчёта'
			),
			array(
				'value' => 'PREPAY_PARTIAL',
				'title' =>
					'Частичная предварительная оплата до момента передачи предмета расчёта'
			),
			array(
				'value' => 'AVANS',
				'title' => 'Аванс'
			),
			array(
				'value' => 'FULL_PAY',
				'title' =>
					'Полная оплата, в том числе с учётом аванса (предварительной оплаты) в момент передачи предмета расчёта'
			),
			array(
				'value' => 'PARTIAL_SETTLEMENT_AND_CREDIT',
				'title' =>
					'Частичная оплата предмета расчёта в момент его передачи с последующей оплатой в кредит'
			),
			array(
				'value' => 'TRANSFER_ON_CREDIT',
				'title' =>
					'Передача предмета расчёта без его оплаты в момент его передачи с последующей оплатой в кредит'
			),
			array(
				'value' => 'CREDIT_PAYMENT',
				'title' =>
					'Оплата предмета расчёта после его передачи с оплатой в кредит (оплата кредита)'
			)
		);

		return $res;
	}

	static function paymentTypeValues()
	{
		$res = array(
			array(
				'value' => 'CASH',
				'title' => 'Сумма оплаты наличными'
			),
			array(
				'value' => 'CARD',
				'title' => 'Сумма оплата безналичными'
			),
			array(
				'value' => 'PREPAY',
				'title' => 'Сумма оплаты предоплатой (зачётом аванса)'
			),
			array(
				'value' => 'POSTPAY',
				'title' => 'Сумма оплаты постоплатой (в кредит)'
			),
			array(
				'value' => 'OTHER',
				'title' => 'Сумма оплаты встречным предоставлением'
			)
		);

		return $res;
	}

	static function vatValues()
	{
		$res = array(
			array(
				'value' => 'WITHOUT_VAT',
				'title' => /*_wp*/('Без НДС')
			),
			array(
				'value' => 'VAT_0',
				'title' => /*_wp*/('НДС 0%')
			),
			array(
				'value' => 'VAT_5',
				'title' => /*_wp*/('НДС 5%')
			),
			array(
				'value' => 'VAT_7',
				'title' => /*_wp*/('НДС 7%')
			),
			array(
				'value' => 'VAT_10',
				'title' => /*_wp*/('НДС 10%')
			),
			array(
				'value' => 'VAT_20',
				'title' => /*_wp*/('НДС 20%')
			),
			array(
				'value' => 'VAT_22',
				'title' => /*_wp*/('НДС 22%')
			),
			array(
				'value' => 'VAT_105',
				'title' => /*_wp*/('НДС 5/105')
			),
			array(
				'value' => 'VAT_107',
				'title' => /*_wp*/('НДС 7/107')
			),
			array(
				'value' => 'VAT_110',
				'title' => /*_wp*/('НДС 10/110')
			),
			array(
				'value' => 'VAT_120',
				'title' => /*_wp*/('НДС 20/120')
			),
			array(
				'value' => 'VAT_122',
				'title' => /*_wp*/('НДС 22/122')
			)
		);

		return $res;
	}

	static function orderPaidValues()
	{
		$res = array(
			array(
				'value' => 0,
				'title' => 'Не учитывать'
			),
			array(
				'value' => 1,
				'title' => 'Нет'
			),
			array(
				'value' => 2,
				'title' => 'Да'
			)
		);

		return $res;
	}

	/**
	 * Возвращает список возможных действий для заказа
	 *
	 * @return array
	 */
	public static function settingAvailableOrderActions()
	{
		$workflow = new shopWorkflow();
		$actions = $workflow->getAvailableActions();
		$data = [];

		foreach ($actions as $key => $action) {
			if (isset($action['state'])) {
				$data[] = [
					'value' => $key,
					'title' => $action['name']
				];
			}
		}

		return $data;
	}

	public static function getDeliveryMethods()
	{
		$spm = new shopPluginModel();
		$methods = $spm->listPlugins('shipping');
		$data = [];

		foreach ($methods as $key => $method) {
			$data[] = [
				'value' => $method['id'],
				'title' => $method['name']
			];
		}

		return $data;
	}

	public static function getPaymentTypes()
	{
		$spm = new shopPluginModel();
		$methods = $spm->listPlugins('payment');
		$data = [];

		foreach ($methods as $k => $v) {
			$data[] = [
				'value' => $v['id'],
				'title' => $v['name']
			];
		}

		return $data;
	}

	public static function getOrganizationsSelect()
	{
		$orgs_sel = self::__createSelects('organization');

		return $orgs_sel;
	}

	public static function getTradeObjectsSelect()
	{
		$trade_objects_sel = self::__createSelects('trade_object');

		return $trade_objects_sel;
	}

	public static function getOrganizationsTextarea()
	{
		$plugin_id = self::PLUGIN_ID;
		$plugin = waSystem::getInstance()->getPlugin($plugin_id, true);
		$login = self::__checkPhone($plugin->getSettings('login'));
		$password = $plugin->getSettings('password');
		$organizations =
			is_array($plugin->getSettings('organizations')) &&
			array_key_first($plugin->getSettings('organizations')) === 0
				? $plugin->getSettings('organizations')
				: [];
		$error = null;
		$errorTxt = '';

		if (!count($organizations) && $login && $password) {
			$tokenData = self::__getToken($login, $password);

			if (
				is_object($tokenData) &&
				isset($tokenData->error) &&
				$tokenData->error != ''
			) {
				$error = 1;
				$errorTxt =
					'<strong>' .
					$tokenData->error .
					'</strong> ' .
					$tokenData->error_description;
			} else {
				$token = $tokenData->access_token;
				$orgData = self::__getOrganizations($token);

				if (
					is_object($orgData) &&
					isset($orgData->error) &&
					$orgData->error != ''
				) {
					$error = 1;
					$errorTxt =
						'<strong>' .
						$orgData->error .
						'</strong> ' .
						$orgData->error_description;
				} else {
					$organizations = self::__createMainArray($orgData, $token);
				}
			}
		}

		if ($error) {
			$controls =
				'<div style="display:inline-block; color:#fff; background:indianred; padding:5px 10px;">' .
				$errorTxt .
				'</div>';
		} else {
			$controls =
				'<textarea id="bifitkassa_shop_bifitkassa_organizations" name="shop_bifitkassa[organizations]" style="display:none;">' .
				json_encode($organizations) .
				'</textarea>';
		}

		$controls .= '<style type="text/css">';
		$controls .= '.article.wider>#wa-plugins-content>h1,.block.double-padded>h1{position:relative}';
		$controls .=
			'.block.double-padded>h1::after{content:\'Касса\'; position:absolute; top:0; left:98px; color:#189AF7; background:#FFF}';
		$controls .= '.article.wider>#wa-plugins-content>h1::after{content:\'Касса\'; position:absolute; top:0; left:124px; color:#189AF7; background:#F3F5FA}';
		$controls .= '.field>.name{text-align:right}';
		$controls .=
			'.field .value input[type="text"], .field .value input[type="password"], .field .value textarea, .field .value select{border:1px solid #AAA; padding:3px}';
		$controls .= '.field .value select{min-width:223px}';
		$controls .= '</style>';
		$controls .= '<script type="text/javascript">';
		$controls .= <<<JS
		function updateLK(){jQuery("#bifitkassa_shop_bifitkassa_organizations").html(""),jQuery("#plugins-settings-form").submit(),location.reload()}function updateTraides(t,s){var o=[],i=jQuery("#bifitkassa_shop_bifitkassa_organizations").html();organizations=jQuery.parseJSON(i),jQuery.each(organizations,function(i,a){if(a.id===t)return jQuery.each(a.traides,function(i,a){o.push("<option value="+a.id+">"+a.name+"</option>")}),jQuery("#bifitkassa_shop_bifitkassa_trade_object").html(o.join("")),s&&jQuery("#up1").text(s),!1})}function updateOrgs(){var t=[],i=jQuery("#bifitkassa_shop_bifitkassa_organizations").html();organizations=jQuery.parseJSON(i),jQuery.each(organizations,function(i,a){t.push("<option value="+a.id+">"+a.name+"</option>")}),jQuery("#bifitkassa_shop_bifitkassa_organization").html(t.join("")),updateTraides(jQuery("#bifitkassa_shop_bifitkassa_organization").find("option").eq(0).val(),"Обновлено!"),jQuery("body,html").animate({scrollTop:jQuery("#up1").offset().top-100},"slow")}jQuery(document).on("change","#bifitkassa_shop_bifitkassa_organization",function(){updateTraides(jQuery(this).val())}),jQuery(document).ready(function(){jQuery("#bifitkassa_shop_bifitkassa_trade_object").has("option").length||(updateOrgs(),jQuery("#plugins-settings-form").submit())});
		JS;
		$controls .= '</script>';

		return $controls;
	}

	static function __saveOrder($order)
	{
		// log
		$log  = '######################' . "\n";
		$log .= '## ПЕРЕДАЧА ЗАКАЗА ###' . "\n";
		$log .= '######################' . "\n";
		$log .= '## ' . date('Y-m-d H:i:s', time()) . "\n";
		$log .= '######################' . "\n";
		$log .= "## " . _wp("Получаем токен...") . "\n";

		// Get token
		$tokenData = self::__getToken($order->login, $order->password);
		$token = $tokenData->access_token;

		// log
		if ($token) {
			$log .= "## " . _wp("Получен Токен") . ": " . $token . "\n";
		} else {
			$log .= "## " . _wp("Ошибка при получении Токена!") . "\n";
			$log .=
				"## " .
				_wp("Ошибка") .
				": " .
				$tokenData->error .
				"\n" .
				"## " .
				_wp("Описание ошибки") .
				": " .
				$tokenData->error_description;

			self::__writeLog($log, $order->number);

			return false;
		}

		// log
		$log .=
			"## " .
			_wp('Проверяем наличие товаров заказа в базе "БИФИТ Касса"...') .
			"\n";

		// parameters for ordering
		$onlineOrderItems = '';
		$onlineOrderPayments = '';

		// form order data
		foreach ($order->products as $key_id => $product) {
			// address for POST request
			$url =
				'https://kassa.bifit.com/cashdesk-api/v1/protected/nomenclatures/external_id/' .
				$product->id .
				'?organization_id=' .
				$order->org_id;

			// link data
			$data = '';

			// headers
			$headers = array(
				'Content-Type: application/x-www-form-urlencoded',
				'Authorization: Bearer ' . $token
			);

			$prod = json_decode(self::__loadData($url, $headers, $data, 'GET'));

			if ($prod == null) {
				// CREATE A NEW PRODUCT

				// log
				$log .=
					"## " .
					_wp("Товар с Id=") .
					$product->id .
					" " .
					_wp('не найден в базе "БИФИТ Касса".') .
					"\n";
				$log .=
					"## " .
					_wp('Добавляем новый товар в базу "БИФИТ Касса"...') .
					"\n";

				// address for POST request
				$url =
					'https://kassa.bifit.com/cashdesk-api/v1/protected/nomenclatures';

				// link data
				$data = '{';
				$data .= '"item": {';
				$data .=
					'"vatValue":' .
					($order->vat_mode == 1
						? $order->vat_value
						: $product->tax_value) .
					',';
				$data .= '"vendorCode": "' . $product->code . '", ';
				$data .= '"name":"' . $product->name . '", ';
				$data .= '"externalId":"' . $product->id . '", ';
				$data .= '"sellingPrice":' . $product->price;
				$data .= '},';
				$data .= '"organizationId":"' . $order->org_id . '"';
				$data .= '}';

				// headers
				$headers = array(
					'Content-Type: application/json',
					'Authorization: Bearer ' . $token
				);

				// log
				$log .=
					_wp('Данные для отправки в базу "БИФИТ Касса"') .
					":" .
					"\n";
				$log .= $data . "\n";

				// get data from request
				$tov = json_decode(self::__loadData($url, $headers, $data));

				// log
				if (!is_object($tov) && !is_array($tov)) {
					$log .=
						"## " .
						_wp(
							'Новый товар успешно добавлен в базу "БИФИТ Касса" с Id='
						) .
						$tov .
						"\n";
				} else {
					$log .=
						"## " .
						_wp(
							'В процессе добавления нового товара произошла ошибка'
						) .
						":";
					$log .= self::__checkResponce($tov) . "\n";
				}
			} else {
				$tov = $prod->id;

				// log
				$log .=
					"## " .
					_wp('Товар с Id=') .
					$product->id .
					" " .
					_wp('уже существует в базе "БИФИТ Касса" с Id=') .
					$tov .
					"\n";
			}

			if (isset($tov) && !is_object($tov) && !is_array($tov)) {
				// add nomenclatureId to product
				$order->products[$key_id]->nomenclature_id = $tov;

				// display product data in the order
				$onlineOrderItems .= $onlineOrderItems != '' ? ',' : '';
				$onlineOrderItems .= '{';
				$onlineOrderItems .= '"nomenclatureId":"' . $tov . '",';
				$onlineOrderItems .= '"paymentSubject":"PRODUCT",';
				$onlineOrderItems .= '"name":"' . $product->name . '",';
				$onlineOrderItems .=
					'"price":' . self::__formatprc($product->price) . ',';
				$onlineOrderItems .=
					'"vatValue":' .
					($order->vat_mode == 1
						? $order->vat_value
						: $product->tax_value) .
					',';
				$onlineOrderItems .= '"quantity":' . $product->count . ',';
				$onlineOrderItems .= '"vendorCode": "' . $product->code . '"';
				$onlineOrderItems .= '}';
			}
		}

		// delivery details
		if ($order->delivery_cost > 0) {
			// address for POST request
			$url =
				"https://kassa.bifit.com/cashdesk-api/v1/protected/nomenclatures/external_id/" .
				"delivery_" .
				$order->delivery_id .
				"?organization_id=" .
				$order->org_id;

			// link data
			$data = false;

			// headers
			$headers = [
				"Content-Type: application/x-www-form-urlencoded",
				"Authorization: Bearer " . $token
			];

			$delivery = json_decode(
				self::__loadData($url, $headers, $data, "GET")
			);

			// log
			$log .=
				"## " .
				_wp(
					'Проверяем наличие услуги "Доставка" в базе "БИФИТ Касса"...'
				) .
				"\n";

			if ($delivery == null) {
				// CREATE A NEW SERVICE

				// log
				$log .=
					"## " .
					_wp('Услуга "Доставка" не найдена в базе "БИФИТ Касса".') .
					"\n";
				$log .=
					"## " .
					_wp(
						'Добавляем новую услугу "Доставка" в базу "БИФИТ Касса"...'
					) .
					"\n";

				// address for POST request
				$url =
					"https://kassa.bifit.com/cashdesk-api/v1/protected/nomenclatures";

				// link data
				$data = "{";
				$data .= '"item":{';
				$data .=
					'"vatValue":' .
					($order->vat_mode == 1 ? $order->vat_value : "0") .
					",";
				$data .= '"name":"' . _wp("Доставка") . '",';
				$data .= '"externalId":"delivery_' . $order->delivery_id . '",';
				$data .= '"sellingPrice":' . $order->delivery_cost;
				$data .= "},";
				$data .= '"organizationId":"' . $order->org_id . '"';
				$data .= "}";

				// headers
				$headers = [
					"Content-Type: application/json",
					"Authorization: Bearer " . $token
				];

				// log
				$log .=
					_wp('Данные для отправки в базу "БИФИТ Касса"') .
					":" .
					"\n";
				$log .= $data . "\n";

				// get data from request
				$tov = json_decode(self::__loadData($url, $headers, $data));

				// log
				if (!is_object($tov) && !is_array($tov)) {
					$log .=
						"## " .
						_wp(
							'Новая услуга "Доставка" успешно добавлена в базу "БИФИТ Касса" с Id='
						) .
						$tov .
						"\n";
				} else {
					$log .=
						"## " .
						_wp(
							'В процессе добавления услуги "Доставка" произошла ошибка:'
						) .
						"\n";
					$log .= self::__checkResponce($tov) . "\n";
				}
			} else {
				$tov = $delivery->id;

				// log
				$log .=
					"## " .
					_wp(
						'Услуга "Доставка" уже существует в базе "БИФИТ Касса" c Id='
					) .
					$tov .
					"\n";
			}

			if (isset($tov) && !is_object($tov) && !is_array($tov)) {
				// add nomenclatureId to delivery
				$order->delivery_nomenclature_id = $tov;

				// display data of delivery method in order
				$onlineOrderItems .= $onlineOrderItems != "" ? "," : "";
				$onlineOrderItems .= "{";
				$onlineOrderItems .=
					'"nomenclatureId":' .
					$order->delivery_nomenclature_id .
					",";
				$onlineOrderItems .= '"paymentSubject":"SERVICE",';
				$onlineOrderItems .= '"name":"' . _wp('Доставка') . '",';
				$onlineOrderItems .= '"price":' . $order->delivery_cost . ",";
				$onlineOrderItems .=
					'"vatValue":' .
					($order->vat_mode == 1 ? $order->vat_value : "0") .
					",";
				$onlineOrderItems .= '"quantity":1';
				$onlineOrderItems .= "}";
			}
		}

		// CREATE A USER IN THE CASH

		// address for POST request
		$url = 'https://kassa.bifit.com/cashdesk-api/v1/protected/clients';

		// headers
		$headers = array(
			'Content-Type: application/json',
			'Authorization: Bearer ' . $token
		);

		// link data
		$data = '{';
		$data .= '"organizationId":"' . $order->org_id . '", ';
		$data .= '"address":"' . $order->user_address . '", ';
		$data .= '"phone":"' . $order->user_phone . '", ';
		$data .= '"lastName":"' . trim($order->user_last_name) . '", ';
		$data .= '"firstName":"' . trim($order->user_name) . '", ';
		$data .= '"email":"' . $order->user_email . '"';
		$data .= '}';

		// log
		$log .=
			"## " .
			_wp('Пробуем добавить нового клиента в базу "БИФИТ Касса"...') .
			"\n";
		$log .=
			"## " .
			_wp('Данные для отправки в базу "БИФИТ Касса"') .
			":" .
			"\n";
		$log .= $data . "\n";

		// get a client
		$client = json_decode(self::__loadData($url, $headers, $data));

		// if there is a client
		if (is_object($client) && $client->type == "ERROR") {
			// address for POST request
			$url =
				'https://kassa.bifit.com/cashdesk-api/v1/protected/clients/list/read?organization_id=' .
				$order->org_id;

			// link data
			$data = false;

			// headers
			$headers = array(
				'Content-Type: application/json',
				'Authorization: Bearer ' . $token
			);

			// get all clients
			$clients = json_decode(self::__loadData($url, $headers, $data));

			foreach ($clients as $item) {
				if ($item->phone == $order->user_phone) {
					$client = $item->id;
					$clientAddress = self::__clearString($item->address);
					$clientLastName = self::__clearString($item->lastName);
					$clientFirstName = self::__clearString($item->firstName);
					$clientPatronymic = $item->patronymic;
					$clientOrganizationInn = $item->organizationInn;
					$clientOrganizationName = $item->organizationName;
					$clientEmail = $item->email;
					$clientComment = self::__clearString($item->comment);
					$clientStatus = $item->status;
					$clientLoyalty = $item->loyalty;
					break;
				}
			}

			$cur =
				'{"address":"' .
				$clientAddress .
				'","lastName":"' .
				$clientLastName .
				'","firstName":"' .
				$clientFirstName .
				'","email":"' .
				$clientEmail .
				'"}';
			$ord =
				'{"address":"' .
				$order->user_address .
				'","lastName":"' .
				$order->user_last_name .
				'","firstName":"' .
				$order->user_name .
				'","email":"' .
				$order->user_email .
				'"}';

			// log
			$log .=
				"## " .
				_wp(
					'Текущий клиент уже существует в базе "БИФИТ Касса" с Id='
				) .
				$client .
				"\n";
			$log .=
				"## " .
				_wp('Проверяем соответствие данных клиента:') .
				"\n" .
				"## " .
				_wp('Данные клиента на сервере:') .
				"\n" .
				$cur .
				"\n" .
				"## " .
				_wp('Данные клиента в заказе:') .
				"\n" .
				$ord .
				"\n";

			if (
				($order->user_email != "" &&
					$clientEmail != $order->user_email) ||
				($order->user_name != "" &&
					$clientFirstName != $order->user_name) ||
				($order->user_last_name != "" &&
					$clientLastName != $order->user_last_name) ||
				($order->user_address != "" &&
					$clientAddress != $order->user_address)
			) {
				$log .=
					"## " .
					_wp(
						'Данные клиента не соответствуют сохранённым, начинаем обновление...'
					) .
					"\n";

				// UPDATE CLIENT DATA

				// address for PUT request
				$url =
					"https://kassa.bifit.com/cashdesk-api/v1/protected/clients/" .
					$client;

				// link data
				$data = "{";
				$data .= '"id":"' . $client . '",';
				$data .= '"organizationId":"' . $order->org_id . '",';
				$data .=
					'"address":"' .
					($order->user_address != "" &&
					$order->user_address != $clientAddress
						? $order->user_address
						: $clientAddress) .
					'",';
				$data .= '"phone":"' . $order->user_phone . '",';
				$data .=
					'"lastName":"' .
					($order->user_last_name != "" &&
					$order->user_last_name != $clientLastName
						? $order->user_last_name
						: $clientLastName) .
					'",';
				$data .=
					'"firstName":"' .
					($order->user_name != "" &&
					$order->user_name != $clientFirstName
						? $order->user_name
						: $clientFirstName) .
					'",';
				$data .= '"patronymic":"' . $clientPatronymic . '",';
				$data .= '"organizationInn":"' . $clientOrganizationInn . '",';
				$data .=
					'"organizationName":"' . $clientOrganizationName . '",';
				$data .=
					'"email":"' .
					($order->user_email != "" &&
					$order->user_email != $clientEmail
						? $order->user_email
						: $clientEmail) .
					'",';
				$data .=
					'"comment":"' .
					($order->user_comment != "" &&
					$order->user_comment != $clientComment
						? $order->user_comment
						: $clientComment) .
					'",';
				$data .= '"status":"' . $clientStatus . '"';
				//$data .= '"loyalty":"' . $clientLoyalty . '"';
				$data .= "}";

				$log .=
					"## " .
					_wp('Данные клиента для обновления:') .
					"\n" .
					$data .
					"\n";

				// headers
				$headers = [
					"Content-Type: application/json",
					"Authorization: Bearer " . $token
				];

				// put data
				$result = json_decode(
					self::__loadData($url, $headers, $data, "PUT")
				);

				if (!is_object($result) && !is_array($result)) {
					$log .=
						"## " .
						_wp('Обновление данных клиента прошло успешно!)') .
						"\n";
				} else {
					$log .=
						"## " .
						_wp('В процессе обновления произошла ошибка(') .
						"\n" .
						self::__checkResponce($result);
				}
			} else {
				$log .=
					"## " .
					_wp('Все данные клиента соответствуют сохраннёным!') .
					"\n";
			}
		} elseif (is_numeric($client)) {
			// log
			$log .=
				"## " .
				_wp('Добавлен новый клиент в базу "БИФИТ Касса" с Id=') .
				$client .
				"\n";
		}

		// SENDING AN ORDER TO THE CASHIER

		// checking order availability
		// address for GET request
		$url =
			'https://kassa.bifit.com/cashdesk-api/v1/protected/online_orders/list/read/' .
			$client .
			'?organization_id=' .
			$order->org_id;

		// link data
		$data = false;

		// headers
		$headers = array(
			'Content-Type: application/x-www-form-urlencoded',
			'Authorization: Bearer ' . $token
		);

		// get orders
		$orders = json_decode(self::__loadData($url, $headers, $data, "GET"));

		if (is_object($orders) && $orders->status == 500) {
			$log .=
				"## " .
				_wp(
					'Ошибка при получении массива заказов из кабинета "БИФИТ Касса"...'
				) .
				"\n" .
				self::__checkResponce($orders);
			self::__writeLog($log, $order->number);

			return false;
		}

		// params
		$order_id = null;
		$delivery_type = "";
		$delivery_time = 0;
		$currentStatusType = "";
		$order_deleted = null;

		// log
		$log .=
			"## " .
			_wp('Проверяем наличие текущего заказа в базе "БИФИТ Касса"...') .
			"\n";

		// trans
		$l = [
			"COURIER" => _wp('Курьер (COURIER)'),
			"SELF" => _wp('Самовывоз (SELF)')
		];

		// search order by id
		foreach ($orders as $key => $item) {
			if ($item->externalId == $order->number) {
				$order_id = $item->id;
				$delivery_type = strtoupper($item->deliveryType);
				$delivery_time = isset($item->deliveryTime)
					? $item->deliveryTime
					: 0;
				$currentStatusType = strtoupper($item->currentStatusType);

				// log
				$log .=
					"## " .
					_wp(
						'Текущий заказ уже существует в базе "БИФИТ Касса" с Id='
					) .
					$order_id .
					". " .
					_wp('Тип доставки=') .
					$l[$delivery_type] .
					"." .
					"\n";

				break;
			}
		}

		// log
		if (!$order_id) {
			$log .=
				"## " .
				_wp('Текущий заказ не найден в базе "БИФИТ Касса".') .
				"\n";
		}

		// type of delivery
		$deliveryType = $delivery_type != "" ? $delivery_type : "SERVICE";

		if (
			isset($order->courier) &&
			in_array($order->delivery_id, $order->courier)
		) {
			$deliveryType = "COURIER";
		} elseif (
			isset($order->self) &&
			in_array($order->delivery_id, $order->self)
		) {
			$deliveryType = "SELF";
		}

		// assign "is_refused"
		$is_refused =
			isset($order_id) &&
			isset($order->refused_statuses) &&
			in_array($order->status_id, $order->refused_statuses) &&
			$order->use_refused
				? 1
				: null;

		if ($deliveryType == "SERVICE") {
			// log
			$txt = !$order_id
				? _wp('Не удалось сохранить текущий заказ')
				: _wp('Не удалось обновить текущий заказ');
			$txt .=
				" " .
				_wp(
					'по причине того что тип доставки заказа не (Курьер или Самовывоз)!'
				);
			$log .= $txt != "" ? "## " . $txt . "\n" : "";
		} elseif (!$order_id || $is_refused) {
			// SAVE || REFUSE

			// address for request
			$url =
				"https://kassa.bifit.com/cashdesk-api/v1/protected/online_orders";

			// headers
			$headers = [
				"Content-Type: application/json",
				"Authorization: Bearer " . $token
			];

			// data
			if ($is_refused) {
				// log
				$log .=
					"## " .
					_wp(
						'Начинаем процесс обновления заказа в базе "БИФИТ Касса"...'
					) .
					"\n";

				$url .= "/" . $order_id . "/status";
				$type = "PUT";
				$statusType = $delivery_type == "SELF" ? "REFUSED" : "DELETED";

				// Basic data
				$data = "{";
				$data .= '"onlineOrderId":"' . $order_id . '",';
				$data .= '"statusType":"' . $statusType . '",';
				$data .= '"statusDate":0,';
				$data .= '"statusInfo":""';
				$data .= "}";
			} elseif (!$order_id) {
				// log
				$log .=
					"## " .
					_wp(
						'Начинаем процесс сохранения нового заказа в базу "БИФИТ Касса"...'
					) .
					"\n";

				// data
				$type = "POST";
				$statusType =
					$deliveryType == "SELF"
						? "ACCEPTED"
						: strtoupper($order->new_status);

				// timestamp
				$timestamp = strtotime($order->created);
				$orderTime =
					strlen($timestamp) == 10 ? $timestamp * 1000 : $timestamp;

				// Basic data
				$onlineOrder = "{";
				$onlineOrder .= '"organizationId":"' . $order->org_id . '",';
				$onlineOrder .= '"tradeObjectId":"' . $order->trade_id . '",';
				$onlineOrder .=
					'"taxSystem":"' . strtoupper($order->tax_system) . '",';
				$onlineOrder .= '"externalId":"' . $order->number . '",';
				$onlineOrder .= '"deliveryType":"' . $deliveryType . '",';
				$onlineOrder .=
					'"deliveryAmount":"' . $order->delivery_cost . '",';
				$onlineOrder .= '"paid":false,';
				$onlineOrder .= '"orderTime":' . $orderTime . ",";
				$onlineOrder .= '"clientId":' . $client . ",";
				$onlineOrder .=
					'"address":"' . addslashes($order->user_address) . '",';
				$onlineOrder .=
					'"comment":"' .
					htmlspecialchars(trim($order->user_comment)) .
					'",';
				$onlineOrder .=
					'"responsiblePersonLogin":"' . $order->login . '",';
				$onlineOrder .= '"totalAmount":' . (float) $order->total . ",";
				$onlineOrder .= '"currentStatusType":"' . $statusType . '",';
				$onlineOrder .= '"currentStatusTime":0';
				$onlineOrder .= "}";

				$data = "{";
				$data .= '"onlineOrder":' . $onlineOrder . ",";
				$data .= '"onlineOrderItems":[' . $onlineOrderItems . "]";
				//$data .= ',"onlineOrderPayments":[]';
				$data .= "}";
			}

			// log
			$log .=
				"## " .
				_wp('Данные заказа для отправки в базу "БИФИТ Касса"') .
				":" .
				"\n";
			$log .= $data . "\n";

			// get the ID of the document being processed (id)
			$order_id = json_decode(
				self::__loadData($url, $headers, $data, $type)
			);

			// log
			if (!is_object($order_id) && !is_array($order_id)) {
				if ($is_refused && $statusType == "DELETED") {
					$order_deleted = true;
					$log .=
						"## " .
						_wp('Заказ успешно удалён из базы "БИФИТ Касса".') .
						$order_id .
						"\n";
				} elseif ($is_refused && $statusType == "REFUSED") {
					$log .=
						"## " .
						_wp('Заказ успешно отменён в базе "БИФИТ Касса".') .
						$order_id .
						"\n";
				} else {
					$log .=
						"## " .
						_wp(
							'Заказ успешно сохранён в базу "БИФИТ Касса" с Id='
						) .
						$order_id .
						"\n";
				}
			} else {
				$log .=
					"## " .
					_wp('В процессе сохранения заказа произошла ошибка') .
					":" .
					"\n";
				$log .= self::__checkResponce($order_id) . "\n";
			}

			// SEND PAYMENT INFORMATION

			if (
				is_numeric($order_id) &&
				!$order_deleted &&
				in_array(strtoupper($order->calc_method), [
					"PREPAY_FULL",
					"PREPAY_PARTIAL",
					"FULL_PAY"
				]) &&
				$order->paid
			) {
				// get order payments
				$orderTransIds = self::__getOrderTransIds(
					$order_id,
					$order->org_id,
					$token
				);

				// params
				$calc_method = strtoupper($order->calc_method);

				// log
				$log .=
					"## " .
					_wp(
						'Начинаем процесс передачи данных по оплате заказа...'
					) .
					"\n";

				// address for request
				$url =
					"https://kassa.bifit.com/cashdesk-api/v1/protected/online_orders/payments";

				// headers
				$headers = [
					"Content-Type: application/json",
					"Authorization: Bearer " . $token
				];

				// data
				$type = "POST";
				$data = "{";
				$data .= '"onlineOrderExternalId":"' . $order->number . '",';

				if (
					!is_object($order_id) &&
					!is_array($order_id) &&
					is_numeric($order_id)
				) {
					$data .= '"onlineOrderId":' . (int) $order_id . ",";
				}

				$data .= '"transactionId":"' . $order->payment_id . '",';
				$data .= '"organizationId":"' . $order->org_id . '",';
				$data .= '"calculationMethod":"' . $calc_method . '",';
				$data .= '"total":' . $order->total;
				$data .= "}";

				// log
				$log .=
					"## " .
					_wp('Данные по оплате для отправки в базу "БИФИТ Касса"') .
					":" .
					"\n";
				$log .= $data . "\n";
				$log .=
					"## " .
					_wp(
						'Проверяем наличие данной оплаты в базе "БИФИТ Касса"...'
					) .
					"\n";

				// check exist
				if (
					count($orderTransIds) &&
					in_array($order->payment_id, $orderTransIds)
				) {
					// log
					$log .=
						"## " .
						_wp('Такая оплата уже есть в базе данных...') .
						"\n";
				} else {
					// log
					$log .=
						"## " .
						_wp('Такой оплаты нет в базе данных, сохраняем...') .
						"\n";

					// get the ID of the document being processed (id)
					$response = json_decode(
						self::__loadData($url, $headers, $data, $type)
					);

					// log
					if (!is_object($response) && !is_array($response)) {
						$log .=
							"## " .
							_wp(
								'Передача данных по оплате прошла успешно. Номер документа='
							) .
							$response .
							"\n";
					} else {
						$log .=
							"## " .
							_wp(
								'В процессе передачи данных по оплате произошла ошибка'
							) .
							":" .
							"\n";
						$log .= self::__checkResponce($response) . "\n";
					}
				}
			}
		} elseif (
			isset($order_id) &&
			!is_object($order_id) &&
			!is_array($order_id) &&
			is_numeric($order_id)
		) {
			// log
			$log .=
				"## " .
				_wp(
					'Начинаем процесс глобального обновления заказа в базе "БИФИТ Касса"...'
				) .
				"\n";

			// address for request
			$url =
				"https://kassa.bifit.com/cashdesk-api/v1/protected/online_orders/" .
				$order_id .
				"/full";

			// headers
			$headers = [
				"Content-Type: application/json",
				"Authorization: Bearer " . $token
			];

			// onlineOrder
			$timestamp = strtotime($order->created);
			$orderTime =
				strlen($timestamp) == 10 ? $timestamp * 1000 : $timestamp;

			$onlineOrder = '"organizationId":"' . $order->org_id . '",';
			$onlineOrder .= '"tradeObjectId":"' . $order->trade_id . '",';
			$onlineOrder .=
				'"taxSystem":"' . strtoupper($order->tax_system) . '",';
			$onlineOrder .= '"externalId":"' . $order->number . '",';
			$onlineOrder .= '"deliveryType":"' . $delivery_type . '",';
			$onlineOrder .=
				'"deliveryAmount":' . (float) $order->delivery_cost . ",";

			if ((int) $delivery_time > 0) {
				$onlineOrder .= '"deliveryTime": ' . $delivery_time . ",";
			}

			$onlineOrder .= '"paid":' . ($order->paid ? "true" : "false") . ",";
			$onlineOrder .= '"orderTime":' . $orderTime . ",";
			$onlineOrder .= '"clientId":' . $client . ",";
			$onlineOrder .=
				'"address":"' . addslashes($order->user_address) . '",';
			$onlineOrder .= '"responsiblePersonLogin":"' . $order->login . '",';
			$onlineOrder .= '"totalAmount":' . (float) $order->total . ",";
			$onlineOrder .=
				'"comment":"' .
				htmlspecialchars(trim($order->user_comment)) .
				'",';
			$onlineOrder .= '"currentStatusType":"' . $currentStatusType . '",';
			$onlineOrder .= '"currentStatusTime":0,';
			$onlineOrder .= '"cashierName":"' . $order->cashier_name . '",';
			$onlineOrder .= '"clientName":"' . $order->user_name . '"';

			// onlineOrderItems
			$onlineOrderItems = "";

			// products
			foreach ($order->products as $key_id => $product) {
				$onlineOrderItems .= $onlineOrderItems != "" ? ", " : "";
				$onlineOrderItems .= "{";
				$onlineOrderItems .=
					'"nomenclatureId":' . $product->nomenclature_id . ",";
				$onlineOrderItems .=
					'"paymentSubject":"' . $product->paymentsubject . '",';
				$onlineOrderItems .= '"name":"' . $product->name . '",';
				$onlineOrderItems .= '"price":' . (float) $product->price . ",";
				$onlineOrderItems .=
					'"vatValue":' .
					($order->vat_mode == 1
						? $order->vat_value
						: $product->tax_value) .
					",";
				$onlineOrderItems .= '"quantity":' . $product->count . ",";
				$onlineOrderItems .= '"vendorCode":"' . $product->code . '"';
				$onlineOrderItems .= "}";
			}

			// delivery
			if ($order->delivery_cost > 0) {
				$onlineOrderItems .= $onlineOrderItems != "" ? "," : "";
				$onlineOrderItems .= "{";
				$onlineOrderItems .=
					'"nomenclatureId":' .
					$order->delivery_nomenclature_id .
					",";
				$onlineOrderItems .= '"paymentSubject":"SERVICE",';
				$onlineOrderItems .=
					'"name":"' . GetMessage("BIFIT_DELIVERY") . '",';
				$onlineOrderItems .=
					'"price":' . (float) $order->delivery_cost . ",";
				$onlineOrderItems .=
					'"vatValue":' .
					($order->vat_mode == 1
						? $order->vat_value
						: $order->delivery_tax_value) .
					",";
				$onlineOrderItems .= '"quantity":1';
				$onlineOrderItems .= "}";
			}

			// onlineOrderPayments
			$onlineOrderPayments = "";

			if (
				in_array(strtoupper($order->calc_method), [
					"PREPAY_FULL",
					"PREPAY_PARTIAL",
					"FULL_PAY"
				]) &&
				$order->paid
			) {
				// get order payments
				$orderTransIds = self::__getOrderTransIds(
					$order_id,
					$order->org_id,
					$token
				);

				// params
				$calc_method = strtoupper($order->calc_method);
				$payments_kassa_id = null;

				if (
					count($orderTransIds) &&
					in_array($order->payment_id, $orderTransIds)
				) {
					$payments_kassa_id = array_search(
						$order->payment_id,
						$orderTransIds
					);
				}

				$onlineOrderPayments .= $onlineOrderPayments != "" ? ", " : "";
				$onlineOrderPayments .= "{";
				$onlineOrderPayments .=
					'"calculationMethod":"' .
					strtoupper($order->calc_method) .
					'",';

				if (
					isset($payments_kassa_id) &&
					!is_object($payments_kassa_id) &&
					!is_array($payments_kassa_id) &&
					is_numeric($payments_kassa_id)
				) {
					$onlineOrderPayments .= '"id":' . $payments_kassa_id . ",";
				}

				$onlineOrderPayments .=
					'"onlineOrderExternalId":"' . $order->number . '",';
				$onlineOrderPayments .=
					'"onlineOrderId":' . (int) $order_id . ",";
				$onlineOrderPayments .=
					'"organizationId":"' . $order->org_id . '",';
				$onlineOrderPayments .=
					'"total":' . (float) $order->total . ",";
				$onlineOrderPayments .=
					'"transactionId":"' . $order->payment_id . '"';
				$onlineOrderPayments .= "}";
			}

			// client
			$client = '"organizationId":"' . $order->org_id . '",';
			$client .=
				'"address":"' .
				($order->user_address != "" &&
				$order->user_address != $clientAddress
					? $order->user_address
					: $clientAddress) .
				'",';
			$client .= '"phone":"' . $order->user_phone . '",';
			$client .=
				'"lastName":"' .
				($order->user_last_name != "" &&
				$order->user_last_name != $clientLastName
					? $order->user_last_name
					: $clientLastName) .
				'",';
			$client .=
				'"firstName":"' .
				($order->user_name != "" &&
				$order->user_name != $clientFirstName
					? $order->user_name
					: $clientFirstName) .
				'",';
			$client .=
				'"email":"' .
				($order->user_email != "" && $order->user_email != $clientEmail
					? $order->user_email
					: $clientEmail) .
				'"';

			// cashierInfos
			$cashierInfos = '"firstName":"' . $order->cashier_name . '",';
			$cashierInfos .=
				'"inn":"' . self::__checkInn($order->cashier_inn, 1) . '"';

			// data
			$data = "{";
			$data .= '"onlineOrder": {' . $onlineOrder . "},";
			$data .= '"onlineOrderItems": [' . $onlineOrderItems . "],";
			$data .= '"onlineOrderPayments": [' . $onlineOrderPayments . "],";
			$data .= '"responsiblePerson": {},';
			$data .= '"client": {' . $client . "},";
			$data .= '"organizationId": "' . $order->org_id . '",';
			$data .= '"cashierInfos": [{' . $cashierInfos . "}]";
			$data .= "}";

			// log
			$log .=
				"## " .
				_wp(
					'Данные заказа для глобального обновления в базе "БИФИТ Касса"'
				) .
				":" .
				"\n";
			$log .= $data . "\n";

			// update order by order_id
			$result = json_decode(
				self::__loadData($url, $headers, $data, "PUT")
			);

			// log
			if (!is_object($result) && !is_array($result)) {
				$log .=
					"## " .
					_wp('Обновление заказа проошло успешно') .
					"!" .
					"\n";
			} else {
				$log .=
					"## " .
					_wp('Во время обновления произошла ошибка:') .
					" " .
					self::__checkResponce($result) .
					"\n";
			}
		}

		// save log
		self::__writeLog($log, $order->number);
	}

	static function __doFiscalize($order)
	{
		// 1. AUTHORIZATION
		//    process necessary to connect a client to fiscal processing

		// address for POST request
		$url =
			'https://fp' .
			$order->test .
			'.bifit.com/processing-api/oauth/token';

		// link data
		$data = 'token=' . $order->token;
		$data .= '&client_id=processing-connector-token';
		$data .= '&client_secret=processing-connector-token';
		$data .= '&grant_type=token';

		// headers
		$headers = array('Content-Type: application/x-www-form-urlencoded');

		// decode the received json response
		$bifit = json_decode(self::__loadData($url, $headers, $data));

		// 2. FORMATION OF THE CONTENT OF A CHECK
		//    formation of information for further sending for fiscalization

		// check parameters
		$items = '';

		// generate check data
		foreach ($order->products as $key_id => $product) {
			// display the product data in the check
			$items .= $items != '' ? ', ' : '';
			$items .= '{';
			$items .=
				'"calculationMethod":"' .
				strtoupper($order->calc_method) .
				'", ';
			$items .= '"paymentSubject":"' . $product->paymentsubject . '", ';
			$items .= '"name":"' . $product->name . '", ';
			$items .= '"price":' . $product->price . ', ';
			$items .= '"quantity":' . $product->count . ', ';
			$items .=
				$order->vat_mode == 1
					? '"vat":"' . strtoupper($order->vat) . '",'
					: '"vat":"' . strtoupper($product->tax) . '",';
			$items .=
				'"total":' .
				self::__formatprc($product->price * $product->count);
			$items .= '}';
		}

		// delivery
		if ($order->delivery_cost > 0) {
			// display the data of the delivery method in the check
			$items .= $items != '' ? ', ' : '';
			$items .= '{';
			$items .=
				'"calculationMethod":"' .
				strtoupper($order->calc_method) .
				'", ';
			$items .= '"paymentSubject":"SERVICE", ';
			$items .= '"name":"' . _wp('Доставка') . '", ';
			$items .=
				'"price":' . self::__formatprc($order->delivery_cost) . ', ';
			$items .= '"quantity":1, ';
			$items .=
				$order->vat_mode == 1
					? '"vat":"' . strtoupper($order->vat) . '",'
					: '"vat":"VAT_0",';
			$items .= '"total":' . self::__formatprc($order->delivery_cost);
			$items .= '}';
		}

		// form a check
		$receipt = '{';
		$receipt .=
			'"type":"' .
			$order->receipt_type .
			'", "taxSystem":"' .
			$order->tax_system .
			'", ';
		$receipt .= '"cashier":{';
		$receipt .= '"name":"' . $order->cashier_name . '", ';
		$receipt .= '"inn":"' . self::__checkInn($order->cashier_inn, 1) . '"';
		$receipt .= '}, ';
		$receipt .= '"client":{';
		$receipt .= $order->fio
			? '"name":"' .
				trim($order->user_name . " " . $order->user_last_name) .
				'", '
			: '';
		$receipt .= '"address":"' . $order->address . '"';
		$receipt .= '}, ';
		$receipt .= '"paymentAddress":"' . $order->payment_address . '", ';
		$receipt .= '"items":[' . $items . '], ';
		$receipt .= '"total":' . self::__formatprc($order->total) . ', ';
		$receipt .=
			'"payments":{"' .
			$order->payment_type .
			'": ' .
			self::__formatprc($order->total) .
			'}';
		$receipt .= '}';

		// 3. SENDING A CHECK TO FISCAL PROCESSING
		//    transfer of the content of the check to fiscal processing for further fiscalization

		// address for POST request
		$url =
			'https://fp' .
			$order->test .
			'.bifit.com/processing-api/protected/documents/registration/receipts';

		// check data
		$data = $receipt;

		// Idempotency-Key
		$o = [];
		$o['number'] = $order->number;
		$o['summ'] = self::__formatprc($order->total);
		$o['cms'] = 'Shop-Script';
		$idemp_key = self::__createGUID($o);

		// headers
		$headers = array(
			'Content-Type: application/json',
			'Authorization: Bearer ' . $bifit->access_token,
			'Idempotency-Key: ' . $idemp_key
		);

		// get the ID of the document being processed (id)
		$receipt_id = json_decode(self::__loadData($url, $headers, $data));

		// 4. RECEIVING DOCUMENT BY ID
		//    obtaining a document with fiscal characteristics by its number

		// options
		$doc = '';

		if (!is_object($receipt_id) && is_numeric($receipt_id)) {
			// link for GET request
			$url =
				'https://fp' .
				$order->test .
				'.bifit.com/processing-api/protected/documents/' .
				$receipt_id;

			// headers
			$headers = array('Authorization: Bearer ' . $bifit->access_token);

			// we get the document
			$doc = json_decode(self::__loadData($url, $headers, false, 'GET'));
		}

		// LINK TO THE CHECK

		// we form a link to the check
		$receipt_link =
			!is_object($receipt_id) && is_numeric($receipt_id)
				? 'https://fp' .
					$order->test .
					'.bifit.com/processing-api/receipts/' .
					$idemp_key
				: 'XXX';

		// OPERATION HISTORY

		// update data
		$order->idemp_key = $idemp_key;
		$order->receipt = $receipt;
		$order->receipt_link = $receipt_link;

		// save the operation to history
		self::__saveReport($order, $receipt_id, $doc, $bifit);
	}

	static function __getOrderData($order)
	{
		// user params
		$user_firstname = $order->contact->get('firstname', 'default');
		$user_lastname = $order->contact->get('lastname', 'default');
		$user_name = trim($user_lastname . ' ' . $user_firstname);
		$user_phone = self::__checkPhone(
			$order->contact->get('phone', 'default')
		);
		$user_email = self::__checkEmail(
			$order->contact->get('email', 'default')
		);
		$user_address = '';
		$address_arr = ['street', 'city', 'region', 'zip', 'country'];

		foreach ($address_arr as $val) {
			$addr = $order->contact->get('address:' . $val, 'default');
			if ($addr != '') {
				$user_address .= $user_address != '' ? ', ' . $addr : $addr;
			}
		}

		// subtotal
		$o_subtotal = 0;
		/// an array of products to fill
		$products = [];
		/// an array of order items
		$basket_items = $order->items;

		// iterate over the array of order items
		foreach ($basket_items as $key_id => $product) {
			/// adjusting the price of the product taking into account discounts
			$p_price = self::__formatprc(
				self::__checkProductPrice(
					$product['price'],
					$order->total,
					$order->discount,
					0
				)
			);

			/// we form a subtotal
			$o_subtotal += $p_price * $product['quantity'];

			// we clear the name of the product from special characters
			$p_name = preg_replace(
				'/[^\p{L}0-9 ]/iu',
				' ',
				html_entity_decode($product['name'])
			);
			$p_name = preg_replace('/ +/', ' ', trim($p_name));

			// form an array of goods for the check
			$products[$key_id] = new stdClass();
			$products[$key_id]->id = $product['product_id'];
			$products[$key_id]->code = $product['sku_id'];
			$products[$key_id]->paymentsubject = "PRODUCT";
			$products[$key_id]->name = $p_name;
			$products[$key_id]->price = $p_price;
			$products[$key_id]->count = (float) $product['quantity'];
			$products[$key_id]->tax_value = (float) $product['tax_percent'];
			$products[$key_id]->tax_included = $product['tax_included'];
			$products[$key_id]->tax =
				(float) $product["tax_percent"] > 0
					? "VAT_" . (int) $product["tax_percent"]
					: "WITHOUT_VAT";
			$products[$key_id]->correction = 0;
		}

		// data
		$data = new stdClass();
		$data->id = $order->id;
		$data->number = (int) filter_var(
			$order->id_str,
			FILTER_SANITIZE_NUMBER_INT
		);
		$data->created = $order->create_datetime;
		$data->changed = $order->update_datetime;
		$data->total = $order->total;
		$data->delivery_id = $order->params['shipping_id'];
		$data->delivery_cost = $order->shipping;
		$data->paid = $order->paid_date ? 1 : null;
		$data->pay_date = $order->paid_date;
		$data->payment_id = $order->params['payment_id'];
		$data->payment_name = $order->params['payment_name'];
		$data->products = $products;
		$data->user_name = $user_firstname;
		$data->user_last_name = $user_lastname;
		$data->user_phone = $user_phone;
		$data->user_email = $user_email;
		$data->user_comment = $order->comment;
		$data->user_address = $user_address;

		return $data;
	}

	public function processOrderStatus($params)
	{
		// ORDER DATA

		/// get order data
		$order = $this->getOrder(ifset($params['order_id']));

		// get data
		$data = self::__getOrderData($order);
		$data->payment_type = $this->getSettings('payment_type');
		$data->payment_address = $this->getSettings('payment_address');
		$data->fio = $this->getSettings('fio');
		$data->address_type = $this->getSettings('address_type');
		$data->address =
			$data->address_type == 'email'
				? $data->user_email
				: $data->user_phone;
		$data->cashier_name = $this->getSettings('cashier_name');
		$data->cashier_inn = $this->getSettings('cashier_inn');
		$data->calc_method = $this->getSettings('calculation_method');
		$data->vat_mode = $this->getSettings('vat_mode');
		$data->vat = $this->getSettings('vat');
		$data->vat_value = str_replace(
			["WITHOUT_VAT", "VAT_"],
			["0", ""],
			strtoupper($data->vat)
		);

		/// payment method
		$payment_id = $data->payment_id;

		/// order status
		$status_id = ifset($params['action_id']);
		$data->status_id = $status_id;

		/// payment State
		$payed = $data->paid ? 1 : null;

		// PLUGINS PARAMS

		/// selected operating mode
		$data->test = $this->getSettings('test') ? '-test' : '';

		/// connection token
		$data->token =
			$data->test != ''
				? 'P5cKbUUD9uSSrSlGdzspLblvBnD0GzTAE0cLmAPSEMxJ79DtLE'
				: $this->getSettings('token');

		// iterating over the document type tabs
		for ($i = 1; $i <= 5; $i++) {
			// will find out whether it is necessary to use this type of document
			$use_type = $this->getSettings('use_type_' . $i) ? true : false;

			// if need to use
			if ($use_type) {
				// get parameters
				/// order paid (yes / no / disregard) (2/1/0)
				$order_paid = $this->getSettings('order_paid_' . $i);
				/// selected payment method (s)
				$pm_ids = $this->getSettings('pm_ids_' . $i);
				/// selected order status (s)
				$pm_statuses = $this->getSettings('pm_statuses_' . $i);
				/// use refused
				$use_refused = $this->getSettings('lk_use_refused') ? 1 : null;
				/// connection tab
				if ($i == 5 && $use_refused) {
					//// selected refused status (s)
					$lk_refused_statuses = $this->getSettings(
						'lk_refused_statuses'
					);
					/// if in array merge status arrays
					if (in_array($status_id, $lk_refused_statuses)) {
						$pm_statuses = array_merge(
							$pm_statuses,
							$lk_refused_statuses
						);
					}
				}

				// update data
				$receipt_types = [
					"1" => "SALE",
					"2" => "SALE_RETURN",
					"3" => "PURCHASE",
					"4" => "PURCHASE_RETURN"
				];
				$data->receipt_type = $receipt_types[$i];
				$data->tax_system = $this->getSettings('tax_system_' . $i);

				// If the payment method and order status match
				if (
					isset($pm_ids) &&
					isset($pm_statuses) &&
					in_array($payment_id, $pm_ids) &&
					in_array($status_id, $pm_statuses) &&
					$data->token != ''
				) {
					// if "order paid" is selected,
					// and it is NOT paid, then we check the following tab
					if ($order_paid == '2' && !$payed) {
						continue;
					}

					// if "order not paid" is selected,
					// and it is paid, then we check the next tab
					if ($order_paid == '1' && $payed) {
						continue;
					}

					// main processes
					if ($i == 5) {
						// Options
						$data->login = self::__checkPhone(
							$this->getSettings('login')
						);
						$data->password = $this->getSettings('password');
						$data->org_id = $this->getSettings('organization');
						$data->trade_id = $this->getSettings('trade_object');
						$data->courier = $this->getSettings(
							'lk_courier_delivery'
						);
						$data->self = $this->getSettings('lk_self_delivery');
						$data->new_status = $this->getSettings(
							'lk_new_courier_order_status'
						);
						$data->use_refused = $this->getSettings(
							'lk_use_refused'
						)
							? 1
							: null;
						$data->refused_statuses = $this->getSettings(
							'lk_refused_statuses'
						);
						// update BIFIT-account data
						self::__saveOrder($data);
					} else {
						// start BIFIT fiscal-process
						self::__doFiscalize($data);
					}
				}
			}
		}
	}
}
