<?php
/*
 * Plugin Name: БИФИТ Касса
 * Plugin URI: www.bifit.com
 * Description: Онлайн-касса
 * Version: 1.2.9
 * Author: BIFIT
 * Author URI: www.bifit.com
 * License: GNU General Public License version 2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 */

require ABSPATH . WPINC . "/pluggable.php";

use Bifit\F\BifitFunctions as bifitF;

final class BifitMainClass
{
	public $version = "1.2.9";

	protected string $test;
	protected string $token;
	protected string $cashier_name;
	protected string $cashier_inn;
	protected string $pay_address;
	protected string $address_type;
	protected string $fio;
	protected string $use_type_1;
	protected string $use_type_2;
	protected string $use_type_3;
	protected string $use_type_4;
	protected string $tax_system_1;
	protected string $tax_system_2;
	protected string $tax_system_3;
	protected string $tax_system_4;
	protected string $calc_method;
	protected string $payment_type;
	protected string $product_vat;
	protected string $shipping_vat;
	protected string $tax_vat;
	protected array $woocommerce_status_1;
	protected array $woocommerce_status_2;
	protected array $woocommerce_status_3;
	protected array $woocommerce_status_4;
	protected array $woocommerce_payment_1;
	protected array $woocommerce_payment_2;
	protected array $woocommerce_payment_3;
	protected array $woocommerce_payment_4;

	private static $_instance = null;

	public static function instance()
	{
		if (is_null(self::$_instance)) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	public function __construct()
	{
		$this->define("BIFIT_ABSPATH", plugin_dir_path(__FILE__));
		$this->define(
			"BIFIT_ABSPATH_VIEWS",
			plugin_dir_path(__FILE__) . "lib/views/"
		);
		$this->includes();
		$this->load_options();
		$this->hooks();
		$this->wp_hooks();
	}

	public function wp_hooks()
	{
		register_activation_hook(__FILE__, ["setup_bifit", "activation"]);
		register_deactivation_hook(__FILE__, ["setup_bifit", "deactivation"]);

		add_action(
			"woocommerce_order_status_changed",
			[$this, "wc_status_changed"],
			10,
			4
		);
	}

	public function hooks()
	{
		add_action("bifit_settings_general", [$this, "settings_general_html"]);
		add_action("bifit_settings_sale", [$this, "settings_sale_html"]);
		add_action("bifit_settings_sale_return", [
			$this,
			"settings_sale_return_html",
		]);
		add_action("bifit_settings_purchase", [
			$this,
			"settings_purchase_html",
		]);
		add_action("bifit_settings_purchase_return", [
			$this,
			"settings_purchase_return_html",
		]);
	}

	public function includes()
	{
		require_once BIFIT_ABSPATH . "lib/setup.php";
		require_once BIFIT_ABSPATH . "lib/functions.php";

		if (is_admin()) {
			require_once BIFIT_ABSPATH . "lib/admin.php";
			add_action("init", ["Bifit_Admin", "init"]);
		}
	}

	private function define($name, $value)
	{
		if (!defined($name)) {
			define($name, $value);
		}
	}

	public function load_options()
	{
		$this->test = get_option("test", 1);
		$this->token = get_option("token", "");
		$this->cashier_name = get_option("cashier_name", "Иванов И.И.");
		$this->cashier_inn = get_option("cashier_inn", "");
		$this->pay_address = get_option("pay_address", "https://site.ru");
		$this->address_type = get_option("address_type", "email");
		$this->fio = get_option("fio", 0);
		$this->use_type_1 = get_option("use_type_1", 0);
		$this->use_type_2 = get_option("use_type_2", 0);
		$this->use_type_3 = get_option("use_type_3", 0);
		$this->use_type_4 = get_option("use_type_4", 0);
		$this->tax_system_1 = get_option("tax_system_1", "COMMON");
		$this->tax_system_2 = get_option("tax_system_2", "COMMON");
		$this->tax_system_3 = get_option("tax_system_3", "COMMON");
		$this->tax_system_4 = get_option("tax_system_4", "COMMON");
		$this->calc_method = get_option("calc_method", "FULL_PAY");
		$this->payment_type = get_option("payment_type", "CARD");
		$this->product_vat = get_option("product_vat", "WITHOUT_VAT");
		$this->shipping_vat = get_option("shipping_vat", "WITHOUT_VAT");
		// added from the side (v.1.2.6)
		$this->tax_vat = get_option("tax_vat", "WITHOUT_VAT");
		// /added from the side (v.1.2.6)
		$this->woocommerce_status_1 = get_option("woocommerce_status_1", []);
		$this->woocommerce_status_2 = get_option("woocommerce_status_2", []);
		$this->woocommerce_status_3 = get_option("woocommerce_status_3", []);
		$this->woocommerce_status_4 = get_option("woocommerce_status_4", []);
		$this->woocommerce_payment_1 = get_option("woocommerce_payment_1", []);
		$this->woocommerce_payment_2 = get_option("woocommerce_payment_2", []);
		$this->woocommerce_payment_3 = get_option("woocommerce_payment_3", []);
		$this->woocommerce_payment_4 = get_option("woocommerce_payment_4", []);

		if (empty($this->token) || $this->test == 1) {
			$this->token = "P5cKbUUD9uSSrSlGdzspLblvBnD0GzTAE0cLmAPSEMxJ79DtLE";
		}
	}

	public function settings_general_html()
	{
		self::load_options(); ?>
		<table class="form-table">
			<tbody>
				<tr>
					<th scope="row">
						<label for="test">Тестовый режим:</label></th>
					<td><select name="test" id="test">
						<?php bifitF::dropdown_list(bifitF::dropdown_yesno_list(), $this->test); ?>
						</select>
						<p class="description">Выберите режим работы плагина...</p></td>
				</tr>
				<tr>
					<th scope="row">
						<label for="token">Токен доступа:</label></th>
					<td><input type="text" name="token" id="token" value="<?php echo esc_attr($this->token); ?>" class="regular-text">
						<p class="description">Введите выданный токен доступа...</p></td>
				</tr>
				<tr>
					<th scope="row">
						<label for="cashier_name">ФИО кассира:</label></th>
					<td><input type="text" name="cashier_name" id="cashier_name" value="<?php echo esc_attr($this->cashier_name); ?>" class="regular-text">
						<p class="description">Введите фамилию и инициалы кассира...</p></td>
				</tr>
				<tr>
					<th scope="row">
						<label for="cashier_inn">ИНН кассира:</label></th>
					<td><input type="text" name="cashier_inn" id="cashier_inn" value="<?php echo esc_attr($this->cashier_inn); ?>" class="regular-text">
						<p class="description">Введите 12-ти значный идентификационный номер налогоплательщика...</p></td>
				</tr>
				<tr>
					<th scope="row">
						<label for="pay_address">Место расчётов:</label></th>
					<td><input type="text" name="pay_address" id="pay_address" value="<?php echo esc_attr($this->pay_address); ?>" class="regular-text">
						<p class="description">Адрес сайта обязательно с HTTP или HTTPS протоколом подлкючения...</p></td>
				</tr>
				<tr>
					<th scope="row"><label for="address_type">Адрес клиента:</label></th>
					<td><select name="address_type" id="address_type">
						<?php bifitF::dropdown_list(bifitF::dropdown_address_type_list(), $this->address_type); ?>
						</select>
						<p class="description">Выберите данные клиента для вывода в чеке...</p></td>
				</tr>
				<tr>
					<th scope="row">
						<label for="fio">ФИО клиента:</label></th>
					<td><select name="fio" id="fio">
						<?php bifitF::dropdown_list(bifitF::dropdown_yesno_list(), $this->fio); ?>
						</select>
						<p class="description">Нужно ли выводить ФИО клиента в чеке?</p></td>
				</tr>
				<tr>
					<th scope="row"><label for="calc_method">Признак способа расчёта:</label></th>
					<td><select name="calc_method" id="calc_method">
						<?php bifitF::dropdown_list(bifitF::dropdown_calc_method_list(), $this->calc_method); ?>
						</select></td>
				</tr>
				<tr>
					<th scope="row"><label for="payment_type">Тип оплаты:</label></th>
					<td><select name="payment_type" id="payment_type">
						<?php bifitF::dropdown_list(bifitF::dropdown_payment_type_list(), $this->payment_type); ?>
						</select></td>
				</tr>
				<tr>
					<th scope="row"><label for="product_vat">НДС на товары:</label></th>
					<td><select name="product_vat" id="product_vat">
						<?php bifitF::dropdown_list(bifitF::dropdown_product_vat_list(), $this->product_vat); ?>
						</select></td>
				</tr>
				<tr>
					<th scope="row"><label for="shipping_vat">НДС на доставку:</label></th>
					<td><select name="shipping_vat" id="shipping_vat">
						<?php bifitF::dropdown_list(bifitF::dropdown_shipping_vat_list(), $this->shipping_vat); ?>
						</select></td>
				</tr>
				<!-- // added from the side (v.1.2.6) -->
				<tr>
					<th scope="row"><label for="tax_vat">НДС на комиссию:</label></th>
					<td><select name="tax_vat" id="tax_vat">
						<?php bifitF::dropdown_list(bifitF::dropdown_tax_vat_list(), $this->tax_vat); ?>
						</select></td>
				</tr>
				<!--// /added from the side (v.1.2.6) -->
			</tbody>
		</table>
		<?php
	}

	public function dropdown_woocommerce_status_list($value)
	{
		$list = wc_get_order_statuses();
		bifitF::dropdown_list($list, $value);
	}

	public function dropdown_woocommerce_payment_list($value)
	{
		$payment_gateways = WC()->payment_gateways->payment_gateways();
		$list = [];

		foreach ($payment_gateways as $gateway):
			$list[$gateway->id] = $gateway->get_title();
		endforeach;

		bifitF::dropdown_list($list, $value);
	}

	public function settings_woocommerce_html($i)
	{
		$woocommerce_status = "woocommerce_status_" . $i;
		$woocommerce_payment = "woocommerce_payment_" . $i;
		?>
		<tr>
			<th scope="row"><label for="<?php echo $woocommerce_status; ?>">Статус оплаты:</label></th>
			<td><select name="<?php echo $woocommerce_status; ?>[]" id="<?php echo $woocommerce_status; ?>" multiple="multiple" size="7">
				<?php $this->dropdown_woocommerce_status_list($this->$woocommerce_status); ?>
				</select>
				<p class="description">Выберите статус(ы) оплаты для активации плагина...</p></td>
		</tr>
		<tr>
			<th scope="row"><label for="<?php echo $woocommerce_payment; ?>">Платёжные системы:</label></th>
			<td><select name="<?php echo $woocommerce_payment; ?>[]" id="<?php echo $woocommerce_payment; ?>" multiple="multiple" size="7">
				<?php $this->dropdown_woocommerce_payment_list($this->$woocommerce_payment); ?>
				</select>
				<p class="description">Выберите способ(ы) оплаты для активации плагина...</p></td>
		</tr>
		<?php
	}

	public function type_tab_html($i)
	{
		$use_type = "use_type_" . $i;
		$tax_system = "tax_system_" . $i;
		?>
		<table class="form-table">
			<tbody>
				<tr>
					<th scope="row">
						<label for="test">Использовать:</label></th>
					<td><select name="<?php echo $use_type; ?>" id="<?php echo $use_type; ?>">
						<?php bifitF::dropdown_list(bifitF::dropdown_yesno_list(), $this->$use_type); ?>
						</select>
						<p class="description">Активация данного типа документа...</p></td>
				</tr>
				<tr>
					<th scope="row"><label for="<?php echo $tax_system; ?>">Система налогообложения:</label></th>
					<td><select name="<?php echo $tax_system; ?>" id="<?php echo $tax_system; ?>">
						<?php bifitF::dropdown_list(bifitF::dropdown_tax_system_list(), $this->$tax_system); ?>
						</select>
						<p class="description">Выберите систему налогообложения...</p></td>
				</tr>
				<?php
				if (function_exists("wc_get_order_statuses")) {
					$this->settings_woocommerce_html($i);
				}?>
			</tbody>
		</table>
		<?php
	}

	public function settings_sale_html()
	{
		$this->type_tab_html(1);
	}

	public function settings_sale_return_html()
	{
		$this->type_tab_html(2);
	}

	public function settings_purchase_html()
	{
		$this->type_tab_html(3);
	}

	public function settings_purchase_return_html()
	{
		$this->type_tab_html(4);
	}

	public static function getProdAttr($product_id, $attr)
	{
		$a = "pa_" . $attr;
		$d = wc_get_product_terms($product_id, $a);

		$slug = $d[0]->slug;
		if ($slug) {
			return $slug;
		}

		return "";
	}

	public static function __createGUID($order)
	{
		// Create GUID (Globally Unique Identifier)
		$guid = "";
		$namespace = rand(11111, 99999);
		$uid = uniqid("", true);
		$data = $namespace;
		$data .= $_SERVER["REQUEST_TIME"];
		$data .= $_SERVER["HTTP_USER_AGENT"];
		$data .= $_SERVER["REMOTE_ADDR"];
		$data .= $_SERVER["REMOTE_PORT"];
		$data .= $order["number"];
		$data .= $order["summ"];
		$hash = strtoupper(hash("ripemd128", $uid . $guid . md5($data)));
		$guid =
			substr($hash, 0, 8) .
			"-" .
			substr($hash, 8, 4) .
			"-" .
			substr($hash, 12, 4) .
			"-" .
			substr($hash, 16, 4) .
			"-" .
			substr($hash, 20, 12);

		return strtolower($order["cms"] . "-" . $guid);
	}

	public function __formatprc($price)
	{
		// конфигурация магазина
		$dec_count = get_option("woocommerce_price_num_decimals", 2);

		// форматируем цену согласно настроек
		$price =
			$dec_count < 0
				? round($price, $dec_count)
				: number_format($price, $dec_count, ".", "");

		return $price;
	}

	public function __checkInn($inn, $type = 1)
	{
		// проверяем заполненность
		if (isset($inn) && trim($inn) != "") {
			if ($type == 2) {
				// только цифры в количестве 10 шт.
				$inn = !preg_match("/^\d{10}$/", trim($inn))
					? "ИНН задан в неверном формате!"
					: trim($inn);
			} elseif ($type == 1) {
				// только цифры в количестве 12 шт.
				$inn = !preg_match("/^\d{12}$/", trim($inn))
					? "ИНН задан в неверном формате!"
					: trim($inn);
			} else {
				// только цифры в количестве 10 или 12 шт.
				$inn = !preg_match("/^\d{10}$|^\d{12}$/", trim($inn))
					? "ИНН задан в неверном формате!"
					: trim($inn);
			}
		} else {
			// ИНН пуст или не найден!
			$inn = "ИНН пуст или не найден!";
		}

		return $inn;
	}

	public function __checkPhone($tel)
	{
		// проверяем заполненность
		if (isset($tel) && trim($tel) != "") {
			// удаляем лишние символы
			$phone = str_replace(["+", "(", ")", "-", " "], "", trim($tel));

			// получаем первую цифру
			$first = substr($phone, "0", 1);

			// если номер из 10 цифр и первая цифра не 7, то добавляем 7
			$phone =
				strlen($phone) == 10 && $first != 7 ? "7" . $phone : $phone;

			// окончательная проверка
			$phone = !preg_match("/^[0-9]{11}+$/", $phone)
				? "Телефон задан в неверном формате!"
				: $phone;
		} else {
			// Телефон пуст или не найден!
			$phone = "Телефон пуст или не найден!";
		}

		return $phone;
	}

	public function __checkEmail($email)
	{
		// проверка модели записи
		$email = !preg_match("/^\+\d+$|^\S+@\S+$/", $email)
			? "E-mail задан в неверном формате!"
			: $email;

		return $email;
	}

	public function __checkProductPrice(
		$item_price,
		$order_subtotal,
		$order_discount = 0
	) {
		// если скидка больше нуля
		if ($order_discount > 0) {
			// вычисляем процент скидки
			$percent = ($order_discount * 100) / $order_subtotal;

			// уменьшаем цену товара согласно скидки
			$item_price = $item_price - $item_price * ($percent / 100);
		}

		return $item_price;
	}

	public function __checkResponce($response)
	{
		$mess = "";

		if (
			is_object($response) &&
			($response->type == "ERROR" || $response->status == 400)
		) {
			$mess = "";

			if ($response->status == 400) {
				$mess .= $response->error . "\n";
				$mess .= $response->message . "\n";
			} else {
				$array = json_decode(json_encode($response), true);
				array_walk_recursive($array, function ($item, $key) use (
					&$mess
				) {
					if ($item != "ERROR" && trim($item != "")) {
						$mess .= $key . ": " . $item . "\n";
					}
				});
			}
		} elseif (is_object($response)) {
			$array = json_decode(json_encode($response), true);
			array_walk_recursive($array, function ($item, $key) use (&$mess) {
				if (trim($item != "")) {
					$mess .= $key . ": " . $item . "\n";
				}
			});
		}

		return $mess;
	}

	public function __loadData(
		$url,
		$headers = false,
		$data = false,
		$type = "POST"
	) {
		$ch = curl_init($url);

		if ($type != "POST") {
			curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $type);
		} else {
			curl_setopt($ch, CURLOPT_POST, true);
		}

		if ($data) {
			curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
		}

		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_ENCODING, "");
		curl_setopt($ch, CURLOPT_MAXREDIRS, 10);
		curl_setopt($ch, CURLOPT_TIMEOUT, 0);
		curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
		curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);

		if ($headers) {
			curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
		}

		$output = curl_exec($ch);

		curl_close($ch);

		return $output;
	}

	public function __saveReport(
		$test,
		$bifit,
		$receipt,
		$receipt_id,
		$doc,
		$object,
		$idemp_key
	) {
		global $wpdb;

		// выводим свойство операции
		$log =
			$test != ""
				? "####### " . "Тестовый запрос" . "\n"
				: "####### " . "Основной запрос" . "\n";

		// устанавливаем часовой пояс согласно настройкам сайта
		date_default_timezone_set(get_option("timezone_string"));
		$log .= "####### " . date("Y-m-d H:i:s", time()) . "\n";

		// ОТВЕТ #1

		$log .= "####### " . "Ответ" . " #1: " . "\n";

		// данные ответа #1
		$token_error = null;

		if (is_object($bifit)) {
			foreach ($bifit as $key => $val) {
				if ($key == "error") {
					$token_error = 1;
				}
				$log .= $key . ": " . $val . "\n";
			}
		}

		// ЧЕК

		// сформированный чек для отправки
		$log .= "####### " . "Чек" . ": " . "\n" . $receipt . "\n";

		if (!$token_error) {
			// КЛЮЧ ИДЕМПОТЕНТНОСТИ

			$log .=
				"####### " .
				"Ключ идемпотентности" .
				": " .
				"\n" .
				$idemp_key .
				"\n";

			// ОТВЕТ #2

			$log .= "####### " . "Ответ" . " #2: " . "\n";

			// данные ответа #2
			if (is_object($receipt_id)) {
				$log .= self::__checkResponce($receipt_id);
			} else {
				$log .= "Номер документа: " . $receipt_id . "\n";

				// ОТВЕТ #3

				$log .= "####### " . "Ответ" . " #3: " . "\n";

				if (is_object($doc)) {
					$log .= self::__checkResponce($doc);
				}

				// ССЫЛКА НА ЧЕК

				// сформированная ссылка на чек
				$log .=
					"####### " .
					"Ссылка на чек" .
					": " .
					"\n" .
					// changed from side
					$object->receipt_link . "\n\n\n" . json_encode($object);
					// /changed from side
			}
		}

		// added from the side (v.1.2.6)
		$date_string = $object->action_date;
		$date = date_create_from_format('Y-m-d H:i:s', $date_string);
		$date->modify('+3 hours');
		// /added from the side (v.1.2.6)

		$object->log = $log;

		// СОХРАНЯЕМ В БД

		$wpdb->insert($wpdb->prefix . "bifit_reports", [
			"order_id" => $object->order_id,
			"order_number" => $object->order_number,
			// changed from the side (v.1.2.6)
			"order_date" => $object->order_date->date('Y-m-d H:i:s'),
			"action_date" => $date->format('Y-m-d H:i:s'),
			// /changed from the side (v.1.2.6)
			"receipt_link" => $object->receipt_link,
			"receipt_id" => $object->receipt_id,
			"receipt_type" => $object->receipt_type,
			"log" => $object->log,
		]);

		// TEMP

		$lastid = $wpdb->insert_id;

		if (isset($lastid) && (int) $lastid > 0) {
			// Save success!
		} else {
			// Saved error!
			$wpdb->show_errors();
			$errorTxt = $wpdb->last_error;

			if (isset($errorTxt) && $errorTxt != '') {
				$file = dirname(__FILE__) . '/log_error.txt';
				$content = file_exists($file) ? file_get_contents($file) : '';
				$content .= '📅 ' . date('Y-m-d H:i:s') . "\n";
				$content .= "———————————————————————" . "\n";
				$content .= 
					is_array($errorTxt) || is_object($errorTxt) 
						? print_r($errorTxt, 1) . "\n"
						: $errorTxt . "\n";
				$content .= "\n";
				file_put_contents($file, $content);
			}

			$wpdb->hide_errors();
		}

		// /temp
	}

	public function __mainProcess($order, $items, $test, $token, $i)
	{
		// 1. АВТОРИЗАЦИЯ
		//    процесс необходимый для подключения клиента к фискальному процессингу

		// адрес для POST-запроса
		$url = "https://fp" . $test . ".bifit.com/processing-api/oauth/token";

		// данные ссылки
		$data = "token=" . $token;
		$data .= "&client_id=processing-connector-token";
		$data .= "&client_secret=processing-connector-token";
		$data .= "&grant_type=token";

		// headers
		$headers = ["Content-Type: application/x-www-form-urlencoded"];

		// декодируем полученный ответ
		$bifit = json_decode(self::__loadData($url, $headers, $data));

		// 2. ФОРМИРОВАНИЕ КОНТЕНТА ЧЕКА
		//    формирование информации для дальнейшей отправки на фискализацию

		// данные заказа
		$order_data = $order->get_data();

		// параметры плагина
		/// имя кассира
		$cashier_name = $this->cashier_name;
		/// ИНН кассира
		$cashier_inn = $this->cashier_inn;
		/// тип документа
		$arr_rec_type = [
			"1" => "SALE",
			"2" => "SALE_RETURN",
			"3" => "PURCHASE",
			"4" => "PURCHASE_RETURN",
		];
		$receipt_type = $arr_rec_type[$i];
		/// система налогов
		$t_s = "tax_system_" . $i;
		$tax_system = $this->$t_s;
		/// тип оплаты
		$payment_type = $this->payment_type;
		/// место расчётов
		$payment_address = $this->pay_address;
		/// тип адреса клиента
		$address_type = $this->address_type;
		/// ФИО клиента
		$fio = $this->fio;
		/// адрес клиента в чеке (телефон или эл.почта)
		if ($address_type == "phone" && $order_data["billing"]["phone"]) {
			$address = self::__checkPhone($order_data["billing"]["phone"]);
		} else {
			$address = self::__checkEmail($order_data["billing"]["email"]);
		}

		// массив товаров для заполнения
		$r_products = [];

		// перебираем массив товаров заказа
		foreach ($items as $key_id => $item) {
			// получаем данные товара
			$item_data = $item->get_data();

			// назначаем НДС
			if ($this->product_vat == 70) {
				$vat = $this->getProdAttr(
					$item->get_product()->get_ID(),
					"stavka_nds"
				);
			} else {
				$vat = $this->product_vat;
			}

			// назначаем признак способа расчёта
			if ($this->calc_method == 70) {
				$calc_method = $this->getProdAttr(
					$item->get_product()->get_ID(),
					"priznak_sposoba_rascheta"
				);
			} else {
				$calc_method = $this->calc_method;
			}

			// цена товара
			$price_piece = round(
				($item_data["total"] + $item_data["total_tax"]) /
					$item_data["quantity"],
				2
			);

			// очищаем имя товара от спец-символов
			$p_name = preg_replace(
				"/[^\p{L}0-9 ]/iu",
				" ",
				html_entity_decode($item_data["name"])
			);
			$p_name = preg_replace("/ +/", " ", trim($p_name));

			// формируем массив товаров для чека
			$r_products[$key_id] = new stdClass();
			$r_products[$key_id]->product_name = $p_name;
			$r_products[$key_id]->product_item_price = $price_piece;
			$r_products[$key_id]->product_quantity = $item_data["quantity"];
			$r_products[$key_id]->product_tax = $vat;
			$r_products[$key_id]->calc_method = $calc_method;
		}

		// параметры для чека
		$items = "";

		// формируем данные чека
		foreach ($r_products as $key_id => $product) {
			// выводим данные товара в чек
			$items .= $items != "" ? ", " : "";
			$items .= "{";
			$items .= '"calculationMethod":"' . $product->calc_method . '", ';
			$items .= '"paymentSubject":"PRODUCT", ';
			$items .= '"name":"' . $product->product_name . '", ';
			$items .= '"price":' . $product->product_item_price . ", ";
			$items .= '"quantity":' . $product->product_quantity . ", ";
			$items .= '"vat":"' . $product->product_tax . '", ';
			$items .=
				'"total":' .
				$product->product_item_price * $product->product_quantity;
			$items .= "}";
		}
		
		// added form the side (v.1.2.6)

		$total_fees = 0;

		foreach ($order->get_fees() as $fee) {
			$total_fees += $fee->get_amount();
		}

		if ($total_fees !== 0) {
			$calc_method_dost = $this->calc_method;

			if ($calc_method_dost == 70) {
				$calc_method_dost = "FULL_PAY";
			}

			// выводим данные способа доставки в чек
			$items .= $items != "" ? ", " : "";
			$items .= "{";
			$items .= '"calculationMethod":"' . $calc_method_dost . '", ';
			$items .= '"paymentSubject":"SERVICE", ';
			$items .= '"name":"Комиссия", ';
			$items .=
				'"price":' .
				self::__formatprc($total_fees) .
				", ";
			$items .= '"quantity":1, ';
			$items .= '"vat":"' . $this->tax_vat . '", ';
			$items .=
				'"total":' .
				self::__formatprc($total_fees);
			$items .= "}";
		}

		// /added from the side (v.1.2.6)

		// доставка
		if ($order_data["shipping_total"] > 0) {
			// назначаем признак способа расчёта
			$calc_method_dost = $this->calc_method;
			if ($calc_method_dost == 70) {
				$calc_method_dost = "FULL_PAY";
			}

			// выводим данные способа доставки в чек
			$items .= $items != "" ? ", " : "";
			$items .= "{";
			$items .= '"calculationMethod":"' . $calc_method_dost . '", ';
			$items .= '"paymentSubject":"SERVICE", ';
			$items .= '"name":"Доставка", ';
			$items .=
				'"price":' .
				self::__formatprc($order_data["shipping_total"]) .
				", ";
			$items .= '"quantity":1, ';
			$items .= '"vat":"' . $this->shipping_vat . '", ';
			$items .=
				'"total":' .
				self::__formatprc(
					$order_data["shipping_total"] + $order_data["shipping_tax"]
				);
			$items .= "}";
		}

		// формируем чек
		$receipt = "{";
		$receipt .=
			'"type":"' .
			$receipt_type .
			'", "taxSystem":"' .
			$tax_system .
			'", ';
		$receipt .= '"cashier":{';
		$receipt .= '"name":"' . $cashier_name . '", ';
		$receipt .= '"inn":"' . self::__checkInn($cashier_inn, 1) . '"';
		$receipt .= "}, ";
		$receipt .= '"client":{';
		$receipt .= $fio
			? '"name":"' .
				trim(
					$order_data["billing"]["last_name"] .
						" " .
						$order_data["billing"]["first_name"]
				) .
				'", '
			: "";
		$receipt .= '"address":"' . $address . '"';
		$receipt .= "}, ";
		$receipt .= '"paymentAddress":"' . $payment_address . '", ';
		$receipt .= '"items":[' . $items . "], ";
		$receipt .= '"total":' . self::__formatprc($order_data["total"]) . ", ";
		$receipt .=
			'"payments":{"' .
			$payment_type .
			'": ' .
			self::__formatprc($order_data["total"]) .
			"}";
		$receipt .= "}";

		// 3. ОТПРАВКА ЧЕКА В ФИСКАЛЬНЫЙ ПРОЦЕССИНГ
		//    передача контента чека в фискальный процессинг для дальнейшей фискализации

		// адрес для POST-запроса
		$url =
			"https://fp" .
			$test .
			".bifit.com/processing-api/protected/documents/registration/receipts";

		// данные чека
		$data = $receipt;

		// ключ идемпотентности (Idempotency-Key)
		$o = [];
		$o["number"] = $order_data["id"];
		$o["summ"] = self::__formatprc($order_data["total"]);
		$o["cms"] = "WooCommerce";
		$idemp_key = self::__createGUID($o);

		// headers
		$headers = [
			"Content-Type: application/json",
			"Authorization: Bearer " . $bifit->access_token,
			"Idempotency-Key: " . $idemp_key,
		];

		// получаем идентификатор документа на процессинге (id)
		$receipt_id = json_decode(self::__loadData($url, $headers, $data));

		// 4. ПОЛУЧЕНИЕ ДОКУМЕНТА ПО ID
		//    получение документа с фискальными признаками по его номеру

		// параметры
		$doc = "";

		if (!is_object($receipt_id)) {
			// ссылка для GET-запроса
			$url =
				"https://fp" .
				$test .
				".bifit.com/processing-api/protected/documents/" .
				$receipt_id;

			// headers
			$headers = ["Authorization: Bearer " . $bifit->access_token];

			// получаем документ
			$doc = json_decode(self::__loadData($url, $headers, false, "GET"));
		}

		// ССЫЛКА НА ЧЕК

		// формируем ссылку на чек
		$receipt_link = !is_object($receipt_id)
			? "https://fp" .
				$test .
				".bifit.com/processing-api/receipts/" .
				$idemp_key
			: "XXX";

		// ИСТОРИЯ ОПЕРАЦИИ

		// параметры объекта
		$object = new stdClass();
		/// id заказа
		$object->order_id = $order_data["id"];
		/// номер заказа
		$object->order_number = $order_data["id"];
		/// дата заказа
		$object->order_date = $order->get_date_created();
		/// дата операции
		date_default_timezone_set(get_option("timezone_string"));
		$object->action_date = date("Y-m-d H:i:s");
		/// ссылка на чек
		$object->receipt_link = $receipt_link;
		/// id полученного документа
		$object->receipt_id = !is_object($receipt_id) ? $receipt_id : "000";
		/// тип проводимого документа
		$object->receipt_type = $receipt_type;
		/// лог операции (будет заполнен далее)
		$object->log = "";
		// сохраняем операцию в историю
		self::__saveReport(
			$test,
			$bifit,
			$receipt,
			$receipt_id,
			$doc,
			$object,
			$idemp_key
		);
	}

	public function wc_status_changed($id, $from, $to, $order)
	{
		global $wpdb;

		// получаем заказ
		$order_data = $order->get_data();

		// данные заказа
		/// статус заказа
		$status_id = "wc-" . $to;
		/// способ оплаты
		$payment_id = $order_data["payment_method"];
		/// товары заказа
		$items = $order_data["line_items"];

		// параметры плагина
		/// выбранный режим работы
		$test = $this->test == 1 ? "-test" : "";
		/// токен для соединения
		$token =
			$test != ""
				? "P5cKbUUD9uSSrSlGdzspLblvBnD0GzTAE0cLmAPSEMxJ79DtLE"
				: $this->token;

		// перебираем вкладки типов документа
		for ($i = 1; $i <= 4; $i++) {
			/// узнаём нужно ли использовать данный тип документа
			$u_t = "use_type_" . $i;
			$use_type = $this->$u_t;
			/// если использовать нужно
			if ($use_type) {
				//// выбранный способ(ы) оплаты
				$wc_p = "woocommerce_payment_" . $i;
				$pm_ids = $this->$wc_p;
				//// выбранный статус(ы) заказа
				$wc_s = "woocommerce_status_" . $i;
				$pm_statuses = $this->$wc_s;
				//// если совпадает способ оплаты и статус заказа
				if (
					isset($pm_ids) &&
					isset($pm_statuses) &&
					in_array($payment_id, $pm_ids) &&
					in_array($status_id, $pm_statuses) &&
					$token != ""
				) {
					///// запускаем БИФИТ-процесс
					self::__mainProcess($order, $items, $test, $token, $i);
				}
			}
		}
	}
}

function Bifit()
{
	return BifitMainClass::instance();
}

$GLOBALS["bifit"] = Bifit();
